(ns burningswell.api.spots
  (:require [burningswell.api.core :as core]
            [burningswell.api.middleware.commands :as commands]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.events :as events]
            [burningswell.api.middleware.authentication :as auth]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.specs :as specs]
            [burningswell.api.validation :as v]
            [burningswell.api.weather :as weather]
            [burningswell.db.spots :as spots]
            [burningswell.db.favourites :as favourites]
            [claro.access :as access]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]))

(s/def :burningswell.api.spots/id
  (s/and ::specs/id (v/db-id-exists? :spots)))

(s/def :burningswell.api.spots/except
  (s/nilable (s/coll-of :burningswell.api.spots/id)))

(s/def :burningswell.api.spots/countries
  (s/nilable (s/coll-of :burningswell.api.countries/id)))

(s/def :burningswell.api.spots/regions
  (s/nilable (s/coll-of :burningswell.api.regions/id)))

(s/def :burningswell.api.spots/params
  (s/keys :opt-un [:burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.search/query
                   :burningswell.api.specs/direction
                   :burningswell.api.specs/distance
                   :burningswell.api.specs/location
                   :burningswell.api.spots/countries
                   :burningswell.api.spots/except
                   :burningswell.api.spots/regions
                   :burningswell.api.spots/sort]))

(defrecord Favourite [id]
  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db user]} spots]
    (favourites/resolve-favourite db user :spot spots)))

;; Create

(s/def :burningswell.api.spots.create.input/location
  (s/keys :req-un [:burningswell.spot/name
                   :burningswell.api.input/location]))

(s/def :burningswell.api.spots.create/input
  (s/keys :req-un [:burningswell.spot/name
                   :burningswell.api.input/location]))

(s/def :burningswell.api.spots/create
  (s/keys :req-un [:burningswell.api.spots.create/input]))

(defrecord Create [input]
  access/Resolve
  (can-resolve? [params {:keys [user]}]
    (auth/is-authenticated user))

  commands/Command
  (command [resolvable env]
    {:name :burningswell.api.commands/create-spot})

  conform/Params
  (conform [params env]
    :burningswell.api.spots/create)

  data/Mutation
  data/Resolvable
  (resolve! [params {:keys [db user]}]
    (->> {:name (:name input)
          :user-id (:id user)
          :location (:location input)}
         (spots/insert! db)))

  events/Events
  (events [resolvable {:keys [user]} spot]
    [{:name :burningswell.api.events/spot-created
      :spot-id (:id spot)
      :user-id (:id user)}]))

;; Spot

(defrecord Spot [id name country region photo]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :spot
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} spots]
    (spots/select-batch db spots)))

;; Spots

(defrecord Spots [after before countries direction distance except
                  first last location regions sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.spots/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:countries countries
          :except except
          :limit (core/limit params)
          :location location
          :offset (core/offset params)
          :query query
          :regions regions
          :sort sort}
         (spots/search db))))

;; Photos

(defrecord Photos [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.photos/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} spots]
    (->> {:join-table :photos-spots
          :limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :photos params)}
         (sql/has-and-belongs-to-many
          db spots :spots :photos))))

(weather/define-weather-resolvers :spots :weather.spots)
