(ns burningswell.api.transform
  (:require [burningswell.api.addresses :as addresses]
            [burningswell.api.airports :as airports]
            [burningswell.api.favourites :as favourites]
            [burningswell.api.continents :as continents]
            [burningswell.api.core :as core]
            [burningswell.api.countries :as countries]
            [burningswell.api.dislikes :as dislikes]
            [burningswell.api.emails :as emails]
            [burningswell.api.flickr.photos :as flickr-photos]
            [burningswell.api.flickr.users :as flickr-users]
            [burningswell.api.likes :as likes]
            [burningswell.api.locations :as locations]
            [burningswell.api.oauth.providers :as oauth-providers]
            [burningswell.api.page-views :as page-views]
            [burningswell.api.pagination :as p]
            [burningswell.api.passwords :as passwords]
            [burningswell.api.photos :as photos]
            [burningswell.api.ports :as ports]
            [burningswell.api.regions :as regions]
            [burningswell.api.roles :as roles]
            [burningswell.api.search :as search]
            [burningswell.api.signin :as signin]
            [burningswell.api.signup :as signup]
            [burningswell.api.slugs :as slugs]
            [burningswell.api.spots :as spots]
            [burningswell.api.time-zones :as time-zones]
            [burningswell.api.users :as users]
            [burningswell.api.weather :as weather]
            [burningswell.api.weather.datasources :as weather-datasources]
            [burningswell.api.weather.models :as weather-models]
            [burningswell.api.weather.variables :as weather-variables]
            [claro.data :as data]
            [clojure.set :as set]
            [clojure.string :as str]))

(defn- assoc-author [m]
  (assoc m :author (users/map->User {:id (:author-id m)})))

(defn- assoc-continent [m]
  (assoc m :continent (continents/map->Continent {:id (:continent-id m)})))

(defn- assoc-time-zone [m]
  (assoc m :time-zone (time-zones/map->TimeZone {:id (:time-zoner-id m)})))

(defn- assoc-country [m]
  (assoc m :country (countries/map->Country {:id (:country-id m)})))

(defn- assoc-spot [m]
  (assoc m :spot (spots/map->Spot {:id (:spot-id m)})))

(defn- assoc-email [m]
  (assoc m :email (emails/map->PrivateEmail {:id (:email-id m)})))

(defn- assoc-location [m]
  (update m :location locations/->Location))

(defn- assoc-user [m]
  (assoc m :user (users/map->User {:id (:user-id m)})))

(defn- assoc-photo [m]
  (->> (when-let [id (:photo-id m)]
         (photos/map->Photo {:id id}))
       (assoc m :photo)))

(defn- assoc-region [m]
  (assoc m :region (regions/map->Region {:id (:region-id m)})))

(defn- assoc-weather-model-datasources [model]
  (assoc model :datasources (weather-models/map->Datasources model)))

(defn- assoc-weather-model-variables [model]
  (assoc model :variables (weather-models/map->Variables model)))

(defn assoc-weather-model [{:keys [model-id] :as m}]
  (assoc m :model (weather-models/map->Model {:id model-id})))

(defn- assoc-weather-variable-models [variable]
  (->> (weather-variables/map->Models variable)
       (assoc variable :models)))

(defn assoc-weather-variable [{:keys [variable-id] :as m}]
  (->> (weather-variables/map->Variable {:id variable-id})
       (assoc m :variable)))

(defn- transform-address
  "Transform the `address`."
  [address]
  (some-> address
          assoc-country
          assoc-location
          assoc-region))

(defn- transform-airport
  "Transform the `airport`."
  [airport]
  (some-> airport
          assoc-country
          assoc-location
          assoc-region))

(defn- transform-continent
  "Transform the `continent`."
  [{:keys [id] :as continent}]
  (some-> continent
          (assoc :favourite (continents/map->Favourite {:id id}))
          (assoc :countries (continents/map->Countries {:id id}))
          (assoc :photos (continents/map->Photos {:id id}))
          (assoc :slug (slugs/->Slug :continent continent))
          (assoc :views (page-views/map->Continent {:id id}))
          (assoc :weather (continents/map->Weather {:id id}))
          assoc-photo))

(defn- transform-country
  "Transform the `country`."
  [{:keys [id] :as country}]
  (some-> country
          (assoc :airports (countries/map->Airports {:id id}))
          (assoc :favourite (countries/map->Favourite {:id id}))
          (assoc :photos (countries/map->Photos {:id id}))
          (assoc :ports (countries/map->Ports {:id id}))
          (assoc :regions (countries/map->Regions {:id id}))
          (assoc :spots (countries/map->Spots {:id id}))
          (assoc :slug (slugs/->Slug :country country))
          (assoc :views (page-views/map->Country {:id id}))
          (assoc :weather (countries/map->Weather {:id id}))
          assoc-continent
          assoc-photo))

(defn- transform-email
  "Transform the `email`."
  [email]
  (some-> email
          assoc-user))

(defn- transform-image
  "Transform the `image`."
  [image]
  (-> image
      (assoc :slug (slugs/->Slug :image image))))

(defn- transform-photo
  "Transform the `photo`."
  [{:keys [id url] :as photo}]
  (some-> photo
          (assoc :flickr (photos/map->Flickr photo))
          (assoc :slug (slugs/->Slug :photo photo))
          (assoc :thumbnail-url (photos/map->ThumbnailUrl photo))
          (assoc :views (page-views/map->Photo photo))))

(defn- transform-photo-like
  "Transform the photo `like`."
  [like]
  (some-> like
          assoc-user
          assoc-photo))

(defn- transform-port
  "Transform the `port`."
  [port]
  (some-> port
          assoc-country
          assoc-location
          assoc-region))

(defn- transform-region
  "Transform the `region`."
  [{:keys [id] :as region}]
  (some-> region
          (assoc :airports (regions/map->Airports {:id id}))
          (assoc :favourite (regions/map->Favourite {:id id}))
          (assoc :photos (regions/map->Photos {:id id}))
          (assoc :ports (regions/map->Ports {:id id}))
          (assoc :spots (regions/map->Spots {:id id}))
          (assoc :slug (slugs/->Slug :region region))
          (assoc :views (page-views/map->Region {:id id}))
          (assoc :weather (regions/map->Weather {:id id}))
          assoc-country
          assoc-photo))

(defmulti transform-search-result :type)

(defmethod transform-search-result :continent [result]
  (continents/map->Continent result))

(defmethod transform-search-result :country [result]
  (countries/map->Country result))

(defmethod transform-search-result :region [result]
  (regions/map->Region result))

(defmethod transform-search-result :spot [result]
  (spots/map->Spot result))

(defmethod transform-search-result :user [result]
  (users/map->User result))

(defn- transform-spot
  "Transform the `spot`."
  [{:keys [address-id id location] :as spot}]
  (some-> spot
          (assoc :address (addresses/map->Address {:id address-id}))
          (assoc :favourite (spots/map->Favourite {:id id}))
          (assoc :location (locations/map->Location {:point location}))
          (assoc :photos (spots/map->Photos {:id id}))
          (assoc :slug (slugs/->Slug :spot spot))
          (assoc :views (page-views/map->Spot {:id id}))
          (assoc :weather (spots/map->Weather {:id id}))
          assoc-country
          assoc-photo
          assoc-user
          assoc-region
          assoc-time-zone))

(defn- transform-time-zone
  "Transform the `time-zone`."
  [time-zone]
  (-> time-zone
      (assoc :slug (slugs/->Slug :time-zone time-zone))))

(defn- transform-user
  "Transform the `user`."
  [{:keys [id] :as user}]
  (some-> user
          (assoc :favourites (users/map->Favourites {:id id}))
          (assoc :emails (users/map->Emails {:id id}))
          (assoc :profile-photos (users/map->ProfilePhotos {:id id}))
          (assoc :roles (users/map->Roles {:id id}))
          (assoc :slug (slugs/->Slug :user user))
          (assoc :views (page-views/map->User {:id id}))
          assoc-country
          assoc-region
          assoc-email))

(defn- transform-weather
  "Transform the weather `datasource`."
  [{:keys [location] :as weather}]
  (let [params {:first 1
                :location
                {:latitude (.getY location)
                 :longitude (.getX location)}}]
    (some-> weather
            (assoc :location (locations/map->Location {:point location}))
            (assoc :wave-heights (weather-variables/map->WaveHeights params))
            (assoc :wind-directions (weather-variables/map->WindDirections
                                     params))
            (assoc :wind-speeds (weather-variables/map->WindSpeeds params)))))

(defn- transform-weather-datasource
  "Transform the weather `datasource`."
  [datasource]
  (some-> datasource
          assoc-weather-model
          (assoc :slug (slugs/->Slug :weather-datasource datasource))))

(defn- transform-weather-model
  "Transform the `weather-model`."
  [model]
  (some-> model
          assoc-weather-model-datasources
          assoc-weather-model-variables
          (assoc :slug (slugs/->Slug :weather-model model))))

(defn- transform-weather-variable
  "Transform the `weather-variable`."
  [variable]
  (some-> variable
          assoc-weather-variable-models
          (assoc :slug (slugs/->Slug :weather-variable variable))))

(defn- transform-wave-height
  "Transform the `wave-height`."
  [wave-height]
  (some-> wave-height
          assoc-location
          assoc-weather-model
          assoc-weather-variable))

(defn- transform-wind-direction
  "Transform the `wind-direction`."
  [wind-direction]
  (some-> wind-direction
          assoc-location
          assoc-weather-model
          assoc-weather-variable))

(defn- transform-wind-speed
  "Transform the `wind-speed`."
  [wind-speed]
  (some-> wind-speed
          assoc-location
          assoc-weather-model
          assoc-weather-variable))

;; Addresses

(extend-protocol data/Transform
  burningswell.api.addresses.Address
  (transform [params address]
    (transform-address address)))

(extend-protocol data/Transform
  burningswell.api.addresses.AddressByLocation
  (transform [params address]
    (transform-address address)))

(extend-protocol data/Transform
  burningswell.api.addresses.Addresses
  (transform [params addresses]
    (p/->Connection addresses addresses/map->Address params)))

;; Airports

(extend-protocol data/Transform
  burningswell.api.airports.Airport
  (transform [params airport]
    (transform-airport airport)))

(extend-protocol data/Transform
  burningswell.api.airports.Airports
  (transform [params airports]
    (p/->Connection airports airports/map->Airport params)))

;; Favourites

(defmulti transform-favourite
  (fn [{:keys [type] :as favourite}] (keyword type)))

(defmethod transform-favourite :continent [favourite]
  (favourites/map->ContinentFavourite favourite))

(defmethod transform-favourite :country [favourite]
  (favourites/map->CountryFavourite favourite))

(defmethod transform-favourite :region [favourite]
  (favourites/map->RegionFavourite favourite))

(defmethod transform-favourite :spot [favourite]
  (favourites/map->SpotFavourite favourite))

(extend-protocol data/Transform
  burningswell.api.favourites.Create
  (transform [params {:keys [type] :as favourite}]
    {:__typename "CreateFavouritePayload"
     :favourite (transform-favourite favourite)}))

(extend-protocol data/Transform
  burningswell.api.favourites.Delete
  (transform [params {:keys [favourite node]}]
    {:__typename "DeleteFavouritePayload"
     :node (assoc node :__typename (some-> favourite :type str/capitalize))}))

(extend-protocol data/Transform
  burningswell.api.favourites.Delete
  (transform [params {:keys [favourite node]}]
    (when favourite
      {:__typename "DeleteFavouritePayload"
       :node
       (case (-> favourite :type keyword)
         :continent (continents/map->Continent node)
         :country (countries/map->Country node)
         :region (regions/map->Region node)
         :spot (spots/map->Spot node))})))

(extend-protocol data/Transform
  burningswell.api.favourites.Favourite
  (transform [params favourite]
    (some-> favourite transform-favourite)))

(extend-protocol data/Transform
  burningswell.api.favourites.ContinentFavourite
  (transform [params favourite]
    (some-> favourite assoc-author assoc-continent)))

(extend-protocol data/Transform
  burningswell.api.favourites.CountryFavourite
  (transform [params favourite]
    (some-> favourite assoc-author assoc-country)))

(extend-protocol data/Transform
  burningswell.api.favourites.RegionFavourite
  (transform [params favourite]
    (some-> favourite assoc-author assoc-region)))

(extend-protocol data/Transform
  burningswell.api.favourites.SpotFavourite
  (transform [params favourite]
    (some-> favourite assoc-author assoc-spot)))

;; Continents

(extend-protocol data/Transform
  burningswell.api.continents.Favourite
  (transform [params favourite]
    (favourites/map->ContinentFavourite favourite)))

(extend-protocol data/Transform
  burningswell.api.continents.Continent
  (transform [params continent]
    (transform-continent continent)))

(extend-protocol data/Transform
  burningswell.api.continents.Continents
  (transform [params continents]
    (p/->Connection continents continents/map->Continent params)))

(extend-protocol data/Transform
  burningswell.api.continents.Countries
  (transform [params countries]
    (p/->Connection countries countries/map->Country params)))

(extend-protocol data/Transform
  burningswell.api.continents.Photos
  (transform [params photos]
    (p/->Connection photos photos/map->Photo params)))

;; (extend-protocol data/Transform
;;   burningswell.api.continents.Weather
;;   (transform [params weather]
;;     (-> weather
;;         (assoc :wave-heights (continents/map->WaveHeights weather)))))

;; (extend-protocol data/Transform
;;   burningswell.api.continents.WaveHeight
;;   (transform [params wave-height]
;;     wave-height))

;; (extend-protocol data/Transform
;;   burningswell.api.continents.WaveHeights
;;   (transform [params wave-heights]
;;     (p/->Connection wave-heights continents/map->WaveHeight params)))

;; Countries

(extend-protocol data/Transform
  burningswell.api.countries.Airports
  (transform [params airports]
    (p/->Connection airports airports/map->Airport params)))

(extend-protocol data/Transform
  burningswell.api.countries.Favourite
  (transform [params favourite]
    (favourites/map->CountryFavourite favourite)))

(extend-protocol data/Transform
  burningswell.api.countries.Country
  (transform [params country]
    (transform-country country)))

(extend-protocol data/Transform
  burningswell.api.countries.Countries
  (transform [params countries]
    (p/->Connection countries countries/map->Country params)))

(extend-protocol data/Transform
  burningswell.api.countries.Photos
  (transform [params photos]
    (p/->Connection photos photos/map->Photo params)))

(extend-protocol data/Transform
  burningswell.api.countries.Ports
  (transform [params ports]
    (p/->Connection ports ports/map->Port params)))

(extend-protocol data/Transform
  burningswell.api.countries.Regions
  (transform [params regions]
    (p/->Connection regions regions/map->Region params)))

(extend-protocol data/Transform
  burningswell.api.countries.Spots
  (transform [params spots]
    (p/->Connection spots spots/map->Spot params)))

;; Dislikes

(extend-protocol data/Transform
  burningswell.api.dislikes.DislikePhoto
  (transform [params dislike]
    (transform-photo-like dislike)))

(extend-protocol data/Transform
  burningswell.api.dislikes.UndislikePhoto
  (transform [params dislike]
    (transform-photo-like dislike)))

;; Emails

(extend-protocol data/Transform
  burningswell.api.emails.PublicEmail
  (transform [params email]
    (assoc (transform-email email) :user nil)))

(extend-protocol data/Transform
  burningswell.api.emails.EmailByAddress
  (transform [params email]
    (transform-email email)))

(extend-protocol data/Transform
  burningswell.api.emails.EmailById
  (transform [params email]
    (transform-email email)))

(extend-protocol data/Transform
  burningswell.api.emails.Emails
  (transform [params emails]
    (p/->Connection emails emails/map->PrivateEmail params)))

;; Flickr Photos

(extend-protocol data/Transform
  burningswell.api.flickr.photos.Photo
  (transform [params {:keys [owner photo-id] :as photo}]
    (-> (assoc photo :owner (flickr-users/map->User {:id owner}))
        (assoc :photo (photos/map->Photo {:id photo-id})))))

;; Likes

(extend-protocol data/Transform
  burningswell.api.likes.LikePhoto
  (transform [params like]
    (transform-photo-like like)))

(extend-protocol data/Transform
  burningswell.api.likes.UnlikePhoto
  (transform [params like]
    (transform-photo-like like)))

;; OAuth Providers

(extend-protocol data/Transform
  burningswell.api.oauth.providers.Provider
  (transform [params provider]
    provider))

(extend-protocol data/Transform
  burningswell.api.oauth.providers.Providers
  (transform [params providers]
    (p/->Connection providers oauth-providers/map->Provider params)))

;; Page Views

(extend-protocol data/Transform
  burningswell.api.page_views.Create
  (transform [params page-view]
    (-> page-view
        (assoc :__typename "PageView")
        (assoc-location)
        (assoc-user))))

;; Passwords

(extend-protocol data/Transform
  burningswell.api.passwords.Reset
  (transform [params email]
    {:__typename "ResetPasswordPayload"
     :email (emails/map->PublicEmail email)}))

;; Photos

(extend-protocol data/Transform
  burningswell.api.photos.Flickr
  (transform [params photo]
    (flickr-photos/map->Photo photo)))

(extend-protocol data/Transform
  burningswell.api.photos.Photo
  (transform [params photo]
    (transform-photo photo)))

(extend-protocol data/Transform
  burningswell.api.photos.Photos
  (transform [params photos]
    (p/->Connection photos photos/map->Photo params)))

;; Ports

(extend-protocol data/Transform
  burningswell.api.ports.Port
  (transform [params port]
    (transform-port port)))

(extend-protocol data/Transform
  burningswell.api.ports.Ports
  (transform [params ports]
    (p/->Connection ports ports/map->Port params)))

;; Regions

(extend-protocol data/Transform
  burningswell.api.regions.Airports
  (transform [params airports]
    (p/->Connection airports airports/map->Airport params)))

(extend-protocol data/Transform
  burningswell.api.regions.Favourite
  (transform [params favourite]
    (favourites/map->RegionFavourite favourite)))

(extend-protocol data/Transform
  burningswell.api.regions.Photos
  (transform [params photos]
    (p/->Connection photos photos/map->Photo params)))

(extend-protocol data/Transform
  burningswell.api.regions.Ports
  (transform [params ports]
    (p/->Connection ports ports/map->Port params)))

(extend-protocol data/Transform
  burningswell.api.regions.Region
  (transform [params region]
    (transform-region region)))

(extend-protocol data/Transform
  burningswell.api.regions.Regions
  (transform [params regions]
    (p/->Connection regions regions/map->Region params)))

(extend-protocol data/Transform
  burningswell.api.regions.Spots
  (transform [params spots]
    (p/->Connection spots spots/map->Spot params)))

;; Roles

(extend-protocol data/Transform
  burningswell.api.roles.Role
  (transform [params role]
    role))

(extend-protocol data/Transform
  burningswell.api.roles.Roles
  (transform [params roles]
    (p/->Connection roles roles/map->Role params)))

;; Search

(extend-protocol data/Transform
  burningswell.api.search.Search
  (transform [params results]
    (p/->Connection results transform-search-result params)))

;; Signin

(extend-protocol data/Transform
  burningswell.api.signin.Create
  (transform [params result]
    (update result :user users/map->User)))

;; Signup

(extend-protocol data/Transform
  burningswell.api.signup.Create
  (transform [params result]
    (update result :user users/map->User)))

;; Spots

(extend-protocol data/Transform
  burningswell.api.spots.Create
  (transform [params spot]
    {:__typename "CreateSpotPayload"
     :spot (spots/map->Spot spot)}))

(extend-protocol data/Transform
  burningswell.api.spots.Favourite
  (transform [params favourite]
    (favourites/map->SpotFavourite favourite)))

(extend-protocol data/Transform
  burningswell.api.spots.Photos
  (transform [params photos]
    (p/->Connection photos photos/map->Photo params)))

(extend-protocol data/Transform
  burningswell.api.spots.Spot
  (transform [params spot]
    (transform-spot spot)))

(extend-protocol data/Transform
  burningswell.api.spots.Spots
  (transform [params spots]
    (p/->Connection spots spots/map->Spot params)))

;; Users

(extend-protocol data/Transform
  burningswell.api.users.Favourites
  (transform [params favourites]
    (p/->Connection favourites favourites/map->Favourite params)))

(extend-protocol data/Transform
  burningswell.api.users.Emails
  (transform [params emails]
    (p/->Connection emails emails/map->Email params)))

(extend-protocol data/Transform
  burningswell.api.users.UserById
  (transform [params user]
    (transform-user user)))

(extend-protocol data/Transform
  burningswell.api.users.UserByUsername
  (transform [params user]
    (transform-user user)))

(extend-protocol data/Transform
  burningswell.api.users.Users
  (transform [params users]
    (p/->Connection users users/map->User params)))

(extend-protocol data/Transform
  burningswell.api.users.Roles
  (transform [params roles]
    (p/->Connection roles roles/map->Role params)))

(extend-protocol data/Transform
  burningswell.api.users.ProfilePhotos
  (transform [params photos]
    (p/->Connection photos photos/map->Photo params)))

;; Time Zones

(extend-protocol data/Transform
  burningswell.api.time_zones.TimeZone
  (transform [params time-zone]
    (transform-time-zone time-zone)))

(extend-protocol data/Transform
  burningswell.api.time_zones.TimeZones
  (transform [params time-zones]
    (p/->Connection time-zones time-zones/map->TimeZone params)))

;; Weather

(extend-protocol data/Transform
  burningswell.api.weather.Weather
  (transform [params weather]
    (transform-weather weather)))

;; Weather Datasources

(extend-protocol data/Transform
  burningswell.api.weather.datasources.Datasource
  (transform [params datasource]
    (transform-weather-datasource datasource)))

(extend-protocol data/Transform
  burningswell.api.weather.datasources.Datasources
  (transform [params datasources]
    (p/->Connection datasources weather-datasources/map->Datasource params)))

;; Weather Models

(extend-protocol data/Transform
  burningswell.api.weather.models.Datasources
  (transform [params datasources]
    (p/->Connection datasources weather-datasources/map->Datasource params)))

(extend-protocol data/Transform
  burningswell.api.weather.models.Model
  (transform [params model]
    (transform-weather-model model)))

(extend-protocol data/Transform
  burningswell.api.weather.models.Models
  (transform [params models]
    (p/->Connection models weather-models/map->Model params)))

(extend-protocol data/Transform
  burningswell.api.weather.models.Variables
  (transform [params variables]
    (p/->Connection variables weather-variables/map->Variable params)))

;; Weather Variables

(extend-protocol data/Transform
  burningswell.api.weather.variables.Variable
  (transform [params variable]
    (transform-weather-variable variable)))

(extend-protocol data/Transform
  burningswell.api.weather.variables.Variables
  (transform [params variables]
    (p/->Connection variables weather-variables/map->Variable params)))

(extend-protocol data/Transform
  burningswell.api.weather.variables.Models
  (transform [params models]
    (p/->Connection models weather-models/map->Model params)))

(extend-protocol data/Transform
  burningswell.api.weather.variables.WaveHeights
  (transform [params wave-heights]
    (p/->Connection wave-heights transform-wave-height params)))

(extend-protocol data/Transform
  burningswell.api.weather.variables.WindDirections
  (transform [params wind-directions]
    (p/->Connection wind-directions transform-wind-direction params)))

(extend-protocol data/Transform
  burningswell.api.weather.variables.WindSpeeds
  (transform [params wind-speeds]
    (p/->Connection wind-speeds transform-wind-speed params)))
