(ns burningswell.api.middleware.authentication
  (:require [buddy.auth.backends :refer [jws]]
            [claro.data :as data]
            [buddy.auth.middleware :as buddy]
            [burningswell.db.roles :as roles]
            [burningswell.db.users :as users]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]))

(defn wrap-jwt-token
  "Wrap `handler` with JWT authentication middleware."
  [handler jwt]
  (->> (jws {:secret (:secret jwt)
             :token-name "Bearer"})
       (buddy/wrap-authentication handler)))

(s/fdef wrap-authentication
  :args (s/cat :handler ifn? :jwt map?))

(defn wrap-user
  "Wrap `handler` with middleware that loads the current user."
  [handler db]
  (fn [{:keys [identity] :as request}]
    (->> (when-let [id (:id identity)]
           (users/by-id db id))
         (assoc request :user)
         (handler))))

(s/fdef wrap-user
  :args (s/cat :handler ifn? :db sql/db?))

(defn wrap-roles
  "Wrap `handler` with middleware that loads the roles of the current user."
  [handler db]
  (fn [{:keys [user] :as request}]
    (if user
      (->> (roles/by-user db user)
           (assoc user :roles)
           (assoc request :user)
           (handler))
      (handler request))))

(s/fdef wrap-roles
  :args (s/cat :handler ifn? :db sql/db?))

(defn wrap-authentication
  "Wrap `handler` with authentication middleware."
  [handler {:keys [db jwt]}]
  (-> handler
      (wrap-roles db)
      (wrap-user db)
      (wrap-jwt-token jwt)))

(defn- roles
  "Returns the roles of `user` as a set of keywords."
  [user]
  (set (map (comp keyword :name) (:roles user))))

(defn has-role?
  "Returns true if the `user` has the `role`."
  [user role]
  (contains? (roles user) role))

(s/fdef has-role?
  :args (s/cat :user (s/nilable map?) :role keyword?))

(defn is-authenticated [user]
  (when-not (:id user)
    (data/error "You must be authenticated.")))
