(ns burningswell.api.page-views
  (:require [burningswell.api.middleware.commands :as commands]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.events :as events]
            [burningswell.api.specs :as specs]
            [burningswell.api.validation :as v]
            [burningswell.db.page-views :as page-views]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [no.en.core :refer [format-url parse-url]]))

;; Input

(s/def :burningswell.api.page-views.create/input
  (s/keys :req-un [:burningswell.api.specs/url
                   :burningswell.api.specs/session-id]
          :opt-un [:burningswell.api.specs/location]))

(s/def :burningswell.api.page-views/create
  (s/keys :req-un [:burningswell.api.page-views.create/input]))

(defn- safe-url
  "Returns a safe `url`."
  [url]
  (some-> (parse-url (str url))
          (update :query-params dissoc :auth-token)
          (format-url)
          (java.net.URL.)))

(defrecord Create [input]
  commands/Command
  (command [params env]
    {:name :burningswell.api.commands/create-page-view
     :params (update params :url safe-url)})

  conform/Params
  (conform [params env]
    :burningswell.api.page-views/create)

  data/Mutation
  data/Resolvable
  (resolve! [params {:keys [user]}]
    (-> (assoc (:input params) :user-id (:id user))
        (update :url safe-url)))

  events/Events
  (events [resolvable {:keys [user]} result]
    [(cond-> {:name :burningswell.api.events/page-view-created
              :session-id (:session-id result)
              :url (-> result :url safe-url)}
       (:location result)
       (assoc :location (:location result))
       user
       (assoc :user-id (:id user)))]))

;; Continent

(s/def :burningswell.api.page-views.continent/viewer-id
  (s/nilable (s/and ::specs/id (v/db-id-exists? :users))))

(s/def :burningswell.api.page-views/continent
  (s/keys :req-un [:burningswell.api.page-views.continent/viewer-id]))

(defrecord Continent [id viewer-id]
  conform/Params
  (conform [params env]
    :burningswell.api.page-views/continent)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} continents]
    (page-views/totals db :continents continents {:id viewer-id})))

;; Country

(s/def :burningswell.api.page-views.country/viewer-id
  (s/nilable (s/and ::specs/id (v/db-id-exists? :users))))

(s/def :burningswell.api.page-views/country
  (s/keys :req-un [:burningswell.api.page-views.country/viewer-id]))

(defrecord Country [id viewer-id]
  conform/Params
  (conform [params env]
    :burningswell.api.page-views/country)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} countries]
    (page-views/totals db :countries countries {:id viewer-id})))

;; Photo

(s/def :burningswell.api.page-views.photo/viewer-id
  (s/nilable (s/and ::specs/id (v/db-id-exists? :users))))

(s/def :burningswell.api.page-views/photo
  (s/keys :req-un [:burningswell.api.page-views.photo/viewer-id]))

(defrecord Photo [id viewer-id]
  conform/Params
  (conform [params env]
    :burningswell.api.page-views/photo)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} photos]
    (page-views/totals db :photos photos {:id viewer-id})))

;; Region

(s/def :burningswell.api.page-views.region/viewer-id
  (s/nilable (s/and ::specs/id (v/db-id-exists? :users))))

(s/def :burningswell.api.page-views/region
  (s/keys :req-un [:burningswell.api.page-views.region/viewer-id]))

(defrecord Region [id viewer-id]
  conform/Params
  (conform [params env]
    :burningswell.api.page-views/region)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} regions]
    (page-views/totals db :regions regions {:id viewer-id})))

;; Spot

(s/def :burningswell.api.page-views.spot/viewer-id
  (s/nilable (s/and ::specs/id (v/db-id-exists? :users))))

(s/def :burningswell.api.page-views/spot
  (s/keys :req-un [:burningswell.api.page-views.spot/viewer-id]))

(defrecord Spot [id viewer-id]
  conform/Params
  (conform [params env]
    :burningswell.api.page-views/spot)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} spots]
    (page-views/totals db :spots spots {:id viewer-id})))

;; User

(s/def :burningswell.api.page-views.user/viewer-id
  (s/nilable (s/and ::specs/id (v/db-id-exists? :users))))

(s/def :burningswell.api.page-views/user
  (s/keys :req-un [:burningswell.api.page-views.user/viewer-id]))

(defrecord User [id viewer-id]
  conform/Params
  (conform [params env]
    :burningswell.api.page-views/user)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} users]
    (page-views/totals db :users users {:id viewer-id})))
