(ns burningswell.api.server
  (:require [aleph.http :as http]
            [burningswell.api.graphql :as graphql]
            [burningswell.api.middleware.authentication :as authentication]
            [burningswell.api.oauth.callback :as oauth-callback]
            [burningswell.api.oauth.connect :as oauth-connect]
            [burningswell.routes :as routes]
            [cheshire.core :as json]
            [clojure.pprint :refer [pprint]]
            [clojure.walk :as walk]
            [com.stuartsierra.component :as component]
            [iapetos.collector.ring :as prometheus]
            [inflections.core :as infl]
            [ring.middleware.cors :refer [wrap-cors]]
            [ring.middleware.format :refer [wrap-restful-format]]
            [ring.middleware.keyword-params :refer [wrap-keyword-params]]
            [ring.middleware.params :refer [wrap-params]]
            [ring.util.io :refer [string-input-stream]]
            [taoensso.timbre :as log]))

(def restful-formats
  "The formats for the `ring-middleware-format` middleware."
  [:json :edn :transit-json :transit-msgpack])

(defmulti route
  "Handle the `request` for the given `route`."
  (fn [env route request]
    (:handler route)))

(defmethod route :graphql [env route request]
  ((graphql/graphql-handler env) request))

(defmethod route :home [env route request]
  ((graphql/graphiql-workspace env) request))

(defmethod route :oauth-callback [env route request]
  (oauth-callback/callback env (merge request route)))

(defmethod route :oauth-connect [env route request]
  (oauth-connect/connect env (merge request route)))

(defmethod route :default [env route request]
  {:status 404 :body "404 - Not found"})

(defn- wrap-debug
  [handler]
  (fn [request]
    (let [body (some-> request :body slurp)
          {:keys [query variables]} (some-> body (json/decode keyword))]
      (log/debug (str "GraphQL Query:\n" query))
      (when-not (empty? variables)
        (log/debug (str "GraphQL Variables\n"
                        (with-out-str (pprint variables)))))
      (handler (assoc request :body (some-> body string-input-stream))))))

(defn- wrap-debug-request
  [handler]
  (fn [request]
    (clojure.pprint/pprint request)
    (handler request)))

(defn- wrap-debug-response
  [handler]
  (fn [request]
    (let [response (handler request)]
      (clojure.pprint/pprint response)
      response)))

(defn- wrap-body-params
  [handler]
  (fn [{:keys [body-params] :as request}]
    (-> (assoc request :body body-params)
        (handler))))

(defn- wrap-transit-response
  [handler]
  (fn [{:keys [headers] :as request}]
    (let [{:keys [body] :as response} (handler request)]
      (if (and (coll? body)
               (#{"application/transit+json"} (get headers "accept")))
        (assoc response :body (infl/hyphenate-keys (walk/keywordize-keys body)))
        response))))

(defn- wrap-exceptions
  [handler]
  (fn [request]
    (try (handler request)
         (catch Exception e
           (log/error e "Request error.")
           (throw e)))))

(defn- wrap-metrics
  "Wrap `handler` with Prometheus metrics."
  [handler {:keys [metrics] :as server}]
  (prometheus/wrap-metrics handler (:registry metrics) {:path "/metrics"}))

(defn handler
  "Returns the Ring handler for the `server`."
  [server]
  (-> (fn [{:keys [uri] :as request}]
        (route server (routes/match uri) request))
      (wrap-transit-response)
      (wrap-body-params)
      (wrap-keyword-params)
      (wrap-params)
      (wrap-restful-format :formats restful-formats)
      ;; (wrap-debug)
      (wrap-cors
       :access-control-allow-credentials "true"
       :access-control-allow-origin [#".*"]
       :access-control-allow-methods [:get :put :post :delete])
      (authentication/wrap-authentication server)
      (wrap-metrics server)
      (wrap-exceptions)))

(defrecord Server [bind-address bind-port]
  component/Lifecycle
  (start [server]
    (->> (http/start-server (handler server) {:port bind-port})
         (assoc server :daemon)))
  (stop [server]
    (some-> server :daemon .close)
    (assoc server :daemon nil)))

(defn server
  "Returns a new server component."
  [config]
  (-> (map->Server config)
      (component/using [:db :client :jwt :metrics :publisher :photos])))
