(ns burningswell.api.specs
  (:require [burningswell.api.core :as core]
            [burningswell.api.validation :as v]
            [burningswell.api.specs.commands]
            [burningswell.api.specs.events]
            [burningswell.specs.core]
            [clojure.edn :as edn]
            [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as gen]
            [clojure.string :as str])
  (:import [org.postgis PGgeometry Point]))

(defn conform!
  "Like `clojure.spec/conform`, but raises an exception if `data`
  doesn't conform to `spec`."
  [spec data]
  (let [result (s/conform spec data)]
    (if (= result ::s/invalid)
      (throw (ex-info "Can't conform data to spec."
                      (s/explain-data spec data)))
      result)))

(s/def distance
  (s/double-in :min 0 :max Double/MAX_VALUE :NaN? false :infinite? false))

(s/def ::min-spots (s/nilable nat-int?))

;; Latitude & Longitude

(s/def ::latitude
  (s/double-in :min -90 :max 90 :NaN? false :infinite? false))

(s/def ::longitude
  (s/double-in :min -180 :max 180 :NaN? false :infinite? false))

;; Ids

(defn- conform-id [x]
  (cond
    (nil? x)
    x
    (number? x)
    x
    (string? x)
    (try (-> x core/base64-decode :id)
         (catch Throwable e ::s/invalid))
    :else ::s/invalid))

(s/def ::id
  (s/conformer conform-id))

;; Location

(defn- conform-point
  "Conform location input."
  [{:keys [longitude latitude] :as location}]
  (cond
    (instance? Point location)
    location
    (and (number? latitude)
         (number? longitude))
    (doto (Point. longitude latitude) (.setSrid 4326))
    :else ::s/invalid))

(s/def :burningswell.api.input/location
  (s/and (s/keys :req-un [::latitude ::longitude])
         (s/conformer conform-point)))

(s/def ::location
  (s/nilable :burningswell.api.input/location))

;; URL

(s/def ::url
  (s/with-gen #(instance? java.net.URL %)
    #(gen/fmap (fn [uri] (java.net.URL. (str uri))) (s/gen uri?))))

(s/def ::uuid uuid?)

(s/def ::session-id uuid?)

;; Pagination

(defn- conform-cursor [x]
  (if (string? x)
    (try (core/base64-decode x)
         (catch Exception _
           ::s/invalid))
    ::s/invalid))

(s/def ::cursor
  (s/and string? (s/conformer conform-cursor)))

(s/def :burningswell.api.pagination/before
  (s/nilable ::cursor))

(s/def :burningswell.api.pagination/after
  (s/nilable ::cursor))

(s/def :burningswell.api.pagination/first
  (s/nilable nat-int?))

(s/def :burningswell.api.pagination/last
  (s/nilable nat-int?))

(s/def :burningswell.api.pagination/params
  (s/keys :req-un [::after ::before ::first ::last]))

;; Search

(s/def :burningswell.api.search/query
  (s/nilable string?))

(s/def :burningswell.api.search/min-spots
  (s/nilable nat-int?))

;; Sorting

(s/def :burningswell.api.specs/direction
  (s/nilable keyword?))

(s/def :burningswell.api.airports/sort
  (s/nilable #{:created-at :distance :name :updated-at}))

(s/def :burningswell.api.continents/sort
  (s/nilable #{:created-at :distance :id :name :updated-at :views}))

(s/def :burningswell.api.countries/sort
  (s/nilable #{:created-at :distance :id :name :spot-count :updated-at :views}))

(s/def :burningswell.api.emails/sort
  (s/nilable #{:address :created-at :id :updated-at}))

(s/def :burningswell.api.favourites/sort
  (s/nilable #{:created-at :updated-at}))

(s/def :burningswell.api.oauth.providers/sort
  (s/nilable #{:created-at :id :name :updated-at}))

(s/def :burningswell.api.photos/sort
  (s/nilable #{:created-at :distance :likes :id :title :updated-at :views}))

(s/def :burningswell.api.ports/sort
  (s/nilable #{:created-at :distance :id :name :type :updated-at :website-url}))

(s/def :burningswell.api.regions/sort
  (s/nilable #{:created-at :distance :id :name :spot-count :updated-at :views}))

(s/def :burningswell.api.roles/sort
  (s/nilable #{:created-at :name :updated-at}))

(s/def :burningswell.api.spots/sort
  (s/nilable #{:created-at :distance :id :name :updated-at :views}))

(s/def :burningswell.api.time-zones/sort
  (s/nilable #{:created-at :id :offset :places :updated-at}))

(s/def :burningswell.api.users/sort
  (s/nilable #{:created-at :distance :first-name :id :last-name
               :updated-at :username :views}))

(s/def :burningswell.api.weather.models/sort
  (s/nilable #{:created-at :id :name :updated-at}))

(s/def :burningswell.api.weather.datasources/sort
  (s/nilable #{:created-at :id :updated-at}))

(s/def :burningswell.api.weather.variables/sort
  (s/nilable #{:created-at :id :name :updated-at}))

(defn boolean-str [s]
  (if (string? s)
    (boolean (Boolean/valueOf s))
    ::s/invalid))

(defn comma-str [s]
  (if (string? s)
    (try (str/split s #"\s*,\s*")
         (catch Exception e
           ::s/invalid))
    ::s/invalid))

(defn re-pattern-str [s]
  (if (string? s)
    (try (re-pattern s)
         (catch Exception e
           ::s/invalid))
    ::s/invalid))

(defn int-comma-str [s]
  (if (string? s)
    (try (mapv #(Integer/parseInt %) (comma-str s))
         (catch Exception e
           ::s/invalid))
    ::s/invalid))

(defn int-str [x]
  (if (integer? x)
    x
    (if (string? x)
      (try (Integer/parseInt x)
           (catch Exception e
             ::s/invalid))
      ::s/invalid)))

(defn keyword-str [s]
  (if (string? s)
    (try (keyword s)
         (catch Exception e
           ::s/invalid))
    ::s/invalid))

(defn string-gen [chars]
  (->> (gen/elements chars)
       (gen/list)
       (gen/fmap (partial apply str))))

(defn comma-list-gen [spec]
  (gen/fmap (partial str/join ",") (gen/not-empty (gen/list (s/gen spec)))))

(def alphanumeric-char?
  (->> (concat (range 48 57)
               (range 65 90)
               (range 97 122))
       (map char)
       (set)))

(defn string-alphanumeric?
  "Returns true if `x` is an alphanumeric string, otherwise false."
  [x]
  (every? alphanumeric-char? x))

(s/def ::string-alphanumeric
  (s/with-gen string-alphanumeric? gen/string-alphanumeric))

(s/def ::string-non-blank
  (s/and string? #(not (str/blank? %))))

(defn keyword-alphanumeric?
  "Returns true if `x` is an alphanumeric string, otherwise false."
  [x]
  (and (keyword x)
       (not (namespace x))
       (string-alphanumeric? (name x))))

(s/def ::keyword-alphanumeric
  (s/with-gen keyword-alphanumeric?
    #(gen/fmap keyword (gen/string-alphanumeric))))

(defn- string-conformer [spec conformer & [str-fn]]
  (s/with-gen (s/and (s/conformer conformer) spec)
    #(gen/fmap (or str-fn str) (s/gen spec))))

(s/def ::boolean-str
  (string-conformer boolean? boolean-str))

(s/def ::pos-int-str
  (s/with-gen (s/and (s/conformer int-str) pos-int?)
    #(gen/fmap str (s/gen pos-int?))))

(s/def ::scheme #{:http :https :postgresql})

(s/def ::scheme-str
  (s/with-gen (s/and (s/conformer keyword-str) ::scheme)
    #(s/gen #{"http" "https"})))

(s/def ::re-pattern
  (s/with-gen #(instance? java.util.regex.Pattern %)
    #(gen/fmap re-pattern (s/gen string?))))

(s/def ::re-pattern-str
  (s/with-gen (s/and (s/conformer re-pattern-str) ::re-pattern)
    #(s/gen string?)))

(s/def ::continent-id
  (s/and ::id (v/db-id-exists? :continents)))

(s/def ::country-id
  (s/and ::id (v/db-id-exists? :countries)))

(s/def ::region-id
  (s/and ::id (v/db-id-exists? :regions)))

(s/def ::spot-id
  (s/and ::id (v/db-id-exists? :spots)))
