(ns burningswell.api.validation
  (:require [burningswell.api.core :refer [*env*]]
            [burningswell.db.favourites :as favourites]
            [burningswell.db.emails :as emails]
            [claro.data :as data]
            [datumbazo.core :as sql]
            [datumbazo.io :refer [citext]]
            [inflections.core :as infl]))

(defn- db [env]
  (or (:db env) (throw (ex-info "Input *env* not bound!" {}))))

(defn- user [env]
  (:user env))

(defn max-length [len]
  #(<= (count %) len))

(defn min-length [len]
  #(>= (count %) len))

(defn email-available? [email-address]
  (emails/available? (db *env*) email-address))

(defn email-exists? [email-address]
  (not (emails/available? (db *env*) email-address)))

(defn email-valid? [email-address]
  ;; TODO: Use apache commons.
  (re-matches #".+@.+" (str email-address)))

(defn username-available? [username]
  (empty? @(sql/select (db *env*) [1]
             (sql/from :users)
             (sql/where `(= :username ~(citext username))))))

(defn db-id-exists?
  "Returns true if the entity of `type` with `id` exists in the
  database, otherwise false."
  [table]
  (fn [id]
    (-> @(sql/select (db *env*) [1]
           (sql/from table)
           (sql/where `(= :id ~id)))
        empty? not)))

(defn not-favourite?
  "Returns true if `input` is no favourite, otherwise false."
  [type]
  (let [foreign-key (favourites/foreign-key type)]
    (fn [input]
      (not (favourites/favourite?
            (db *env*) (user *env*) type
            {:id (:id input)})))))

(defn is-favourite-author
  [db favourite author]
  (when-not (favourites/author? db favourite author)
    (data/error "You must be the author of the favourite.")))
