(ns burningswell.api.weather.models
  (:require [burningswell.api.core :as core]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.db.util :refer [fulltext]]
            [burningswell.db.weather.models :as models]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]))

(s/def :burningswell.api.weather.models/params
  (s/keys :opt-un [:burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.search/query
                   :burningswell.api.specs/direction
                   :burningswell.api.weather.models/sort]))

(defrecord Model [id name description]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :weather-model
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} models]
    (models/select-batch db models)))

(defrecord Models [after before direction first last sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.weather.models/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    @(sql/select db [:models.id]
       (sql/from :weather.models)
       (fulltext query :name :description)
       (sql/limit (core/limit params))
       (sql/offset (core/offset params))
       (sql/order-by (keyword (or sort :name))))))

(defrecord Datasources [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.weather.datasources/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} models]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by '(order-by :datasources.model-id
                               (desc :datasources.reference-time))}
         (sql/has-many
          db models :weather.models :weather.datasources))))

(defrecord Variables [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.weather.variables/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} models]
    (->> {:join-table :weather.models-variables
          :limit (core/limit params)
          :offset (core/offset params)
          :order-by '(order-by :variables.name)}
         (sql/has-and-belongs-to-many
          db models :weather.models :weather.variables))))
