(ns burningswell.api.ratings
  "The ratings in the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.ratings :as ratings]
            [burningswell.http.response :refer [created ok]]
            [datumbazo.core :refer [with-connection]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn rating-not-found
  "Return a 404 response for a rating that could not be found by `id`."
  [id]
  (not-found (format "Rating %s not found" id)))

(defn ratings
  "List all ratings."
  [{:keys [api-client db query-params]}]
  (with-connection [db db]
    (let [ratings (ratings/ratings db query-params)]
      (ok ratings))))

(defn create-rating
  "Create a new rating."
  [{:keys [api-client data broker db identity path-params]}]
  (with-connection [db db]
    (s/validate CreateRating data)
    (let [rating (ratings/create-rating
                  db (assoc data :user-id (:id identity)))]
      (publish broker "ratings.created" rating)
      (created rating))))

(defn rating
  "Show a rating."
  [{:keys [api-client db path-params]}]
  (with-connection [db db]
    (if-let [rating (ratings/rating-by-id db (:id path-params))]
      (ok rating)
      (rating-not-found (:id path-params)))))

(defn delete-rating
  "Delete a rating."
  [{:keys [broker db path-params]}]
  (with-connection [db db]
    (if-let [rating (ratings/rating-by-id db (:id path-params))]
      (do (ratings/delete-rating db rating)
          (publish broker "ratings.deleted" rating)
          (no-content))
      (rating-not-found (:id path-params)))))

(defn update-rating
  "Update a rating."
  [{:keys [api-client data broker db path-params]}]
  (with-connection [db db]
    (s/validate CreateRating data)
    (if-let [rating (ratings/rating-by-id db (:id path-params))]
      (let [rating (ratings/save-rating db (merge rating data))]
        (publish broker "ratings.updated" rating)
        (ok rating))
      (rating-not-found (:id path-params)))))

(set! *warn-on-reflection* false)
