(ns burningswell.api.facebook
  (:require [burningswell.api.jwt :as jwt]
            [burningswell.db.users :as users]
            [clj-http.client :as http]
            [inflections.core :as inf]
            [no.en.core :refer [format-url]]
            [ring.util.response :as ring]
            [slingshot.slingshot :refer [try+]]))

(def access-token-url
  "The Facebook access token endpoint url."
  "https://graph.facebook.com/v2.3/oauth/access_token")

(def me-url
  "The Facebook me endpoint url."
  "https://graph.facebook.com/v2.3/me")

(def unknown-error
  {:error :unknown-error
   :description "Something went wrong logging in with Facebook."})

(defmulti callback-error
  "Translate an error received by the callback."
  (fn [error]
    (keyword (inf/hyphenate (:error_reason error)))))

(defmethod callback-error :user-denied [error]
  {:error :user-denied
   :description "You denied logging in with Facebook."})

(defmethod callback-error :default [error]
  unknown-error)

(defmulti request-error
  "Translate an error received by a request."
  (fn [error] (:code error)))

(defmethod request-error 1 [error]
  {:error :invalid-client-secret
   :description (:message error)})

(defmethod request-error 100 [error]
  {:error :auth-code-expired
   :description "This authorization code has expired."})

(defmethod request-error 101 [error]
  {:error :invalid-client-id
   :description (:message error)})

(defmethod request-error 191 [error]
  {:error :invalid-redirect-uri
   :description "Invalid redirect URI."})

(defmethod request-error :default [error]
  unknown-error)

(defn redirect-url
  "Return the redirect url to the Burning Swell web app."
  [web query-params]
  (->> (inf/stringify-keys query-params)
       (inf/stringify-values)
       (inf/hyphenate-keys)
       (assoc web :uri "/signin" :query-params)
       (format-url)))

(defn request-access-token
  "Request an access token for `code`."
  [facebook code]
  (->> {:as :json
        :coerce :always
        :query-params
        {:client_id (:client-id facebook)
         :client_secret (:client-secret facebook)
         :code code
         :redirect_uri (:redirect-uri facebook)}}
       (http/get access-token-url)
       :body inf/hyphenate-keys))

(defn request-facebook-user
  "Request the current user for `oauth-token`."
  [facebook access-token]
  (->> {:as :json
        :coerce :always
        :oauth-token (:access-token access-token)}
       (http/get me-url)
       :body inf/hyphenate-keys))

(defn handle-code
  "Exchange `code` against an access token, save the user to the
  database and redirect to the web app with a JWT token in the query
  params."
  [db facebook jwt web code]
  (->> (request-access-token facebook code)
       (request-facebook-user facebook)
       (users/save-facebook-user db)
       (jwt/jwt-token jwt)
       (redirect-url web)))

(defn login
  "The Facebook login callback endpoint."
  [{:keys [db config query-params]}]
  (let [{:keys [ facebook jwt web]} config]
    (ring/redirect
     (try+
      (cond
        (:code query-params)
        (handle-code db facebook jwt web (:code query-params))
        (:error query-params)
        (redirect-url web (callback-error query-params)))
      (catch [:status 400] {:keys [body]}
        (redirect-url web (request-error (:error body))))))))
