(ns burningswell.api.coerce
  "Schema input and output coercion."
  (:require [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [clj-time.coerce :refer [to-date]]
            [geo.postgis :as geo]
            [no.en.core :refer [parse-double parse-long]]
            [inkspot.color :as color]
            [schema.coerce :as coerce]
            [schema.utils :refer [error-val]]
            [schema.core :as s])
  (:import [java.awt Color]
           [org.postgis Point]))

(def coercion-matcher
  (coerce/first-matcher
   [coerce/string-coercion-matcher
    {s/Inst to-date}]))

(defn coerce
  "Coerce `data` into `schema`."
  [schema data]
  (let [result ((coerce/coercer schema coercion-matcher) data)]
    (if-let [error (error-val result)]
      (throw (ex-info "Coercion error"
                      {:type :coercion-error
                       :schema schema
                       :data data}))
      result)))

(defn update-color [m]
  (update-in
   m [:color]
   (fn [color]
     (cond
       (nil? color)
       color
       (re-matches #"(?i)[0-9A-F]{6}" (str color))
       (color/coerce (str "#" color))
       :else (color/coerce color)))))

(defn update-zoom [m]
  (update-in
   m [:zoom]
   (fn [zoom]
     (cond
       (nil? zoom)
       nil
       (re-matches #"[0-9]" (str zoom))
       (Long/parseLong (str zoom))
       :else
       (throw (ex-info "Invalid zoom level" {:zoom zoom}))))))

(defn update-bounding-box
  "Assoc a :bounding-box onto `m` if it has :top, :right, :bottom
  and :left keys."
  [m]
  (if (and (:top m) (:right m) (:bottom m) (:left m))
    (try (assoc m :bounding-box
                (geo/bounding-box
                 (geo/point
                  4326
                  (Double/parseDouble (:left m))
                  (Double/parseDouble (:bottom m)) )
                 (geo/point
                  4326
                  (Double/parseDouble (:right m))
                  (Double/parseDouble (:top m)))))
         (catch Exception e
           (throw (ex-info "Invalid bounding box"
                           (select-keys m [:top :right :bottom :left])))))
    m))

(defn update-location
  "Convert :latitude and :longitude coordinates into a PostGIS point,
  assoc the point under the :location key and remove the :latitude
  and :longitude keys."
  [m]
  (let [{:keys [latitude longitude]} m]
    (if (and latitude longitude)
      (try (let [latitude (Double/parseDouble latitude)
                 longitude (Double/parseDouble longitude)]
             (assoc m :location (geo/point 4326 longitude latitude)))
           (catch Exception e
             (throw (ex-info "Invalid location"
                             (select-keys m [:latitude :longitude])))))
      m)))

(defn update-page
  "Coerce the :page parameter in `m` to a number."
  [m]
  (update-in m [:page] #(or (parse-long %) 1)))

(defn update-per-page
  "Coerce the :per-page parameter in `m` to a number."
  [m & [per-page]]
  (update-in m [:per-page] #(or (parse-long %) per-page 10)))

(defn update-pagination
  "Coerce the pagination parameters in `m` to numbers."
  [m & [per-page]]
  (-> (update-page m)
      (update-per-page per-page)))

(defn update-timestamp
  "Coerce the value of `k` in `m` into a date."
  [m k]
  (if-let [value (get m k)]
    (assoc m k (to-date value))
    m))

;; Input coercion

(defmulti input (fn [schema] schema))

(defmethod input Color [schema]
  (fn [request color]
    (cond
      (re-matches #"(?i)[0-9A-F]{6}" (str color))
      (color/coerce (str "#" color))
      :else (color/coerce color))))

(defmethod input Id [schema]
  (fn [request id]
    (or (parse-long id) id)))

(defmethod input CountriesParams [schema]
  (fn [request params]
    (-> (update-location params)
        (update-pagination))))

(defmethod input PaginationParams [schema]
  (fn [request params]
    (update-pagination params)))

(defmethod input SearchParams [schema]
  (fn [request params]
    (-> (update-location params)
        (update-pagination 5))))

(defmethod input SpotsParams [schema]
  (fn [request params]
    (-> (coerce SpotsParams params)
        (update-bounding-box)
        (update-location)
        (update-pagination))))

(defmethod input RegionsParams [schema]
  (fn [request params]
    (-> (update-location params)
        (update-pagination))))

(defmethod input WaveHeightParams [schema]
  (fn [request params]
    (-> (update-pagination params)
        (update-timestamp :start)
        (update-timestamp :end))))

(defmethod input WeatherParams [schema]
  (fn [request params]
    (-> (update-pagination params)
        (update-timestamp :start)
        (update-timestamp :end))))

(defmethod input :default [schema])

;; Output coercion

(defmulti output (fn [schema] schema))

(defmethod output :default [schema])

(defmethod output EmbeddedTimeZone [schema]
  (fn [request time-zone]
    (hal/link request :time-zone time-zone)))

(defmethod output TimeZone [schema]
  (fn [request time-zone]
    (hal/link request :time-zone time-zone)))
