(ns burningswell.api.users
  "The Burning Swell users."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.validation :as validation]
            [burningswell.db.roles :as roles]
            [burningswell.db.users :as users]
            [burningswell.http.response :refer [created ok]]
            [datumbazo.core :refer [with-connection]]
            [schema.core :as s]))

(defn user-not-found
  "Return a 404 response for a user that could not be found by `id`."
  [id]
  (not-found (format "User %s not found" id)))

(defn me
  "Return the currently logged in user."
  [{:keys [api-client db identity]}]
  (if-let [user (users/by-id db (:id identity))]
    (ok (hal/link api-client :user user))))

(defn users
  "List all users."
  [{:keys [api-client db query-params]}]
  (->> (users/all db query-params)
       (map #(users/safe-user identity %))
       (hal/links api-client :user)
       (ok)))

(defn create-user
  "Create a new user."
  [{:keys [api-client data broker db]}]
  (validation/validate-errors! {:db db} ::validation/signup data)
  (with-connection [db db]
    (let [user (users/insert db data)
          user (hal/link api-client :user user)]
      (users/add-to-role db user (roles/surfer db))
      (publish broker "users.created" user)
      (created user))))

(defn user
  "Show a user."
  [{:keys [api-client db identity path-params] :as request}]
  (if (and (:id identity) (= (:id path-params) "me"))
    (me request)
    (if-let [user (users/find db (:id path-params))]
      (->> (users/safe-user identity user)
           (hal/link api-client :user)
           (ok))
      (user-not-found (:id path-params)))))

(defn delete-user
  "Delete a user."
  [{:keys [broker db path-params]}]
  (if-let [user (users/by-id db (:id path-params))]
    (do (users/delete db user)
        (publish broker "users.deleted" user)
        (no-content))
    (user-not-found (:id path-params))))

(defn update-user
  "Update a user."
  [{:keys [api-client data broker db path-params]}]
  (if-let [user (users/by-id db (:id path-params))]
    (do (s/validate (CreateUser db) data)
        (let [user (users/update db (merge user data))
              user (hal/link api-client :user user)]
          (publish broker "users.updated" user)
          (ok user)))
    (user-not-found (:id path-params))))

(defn validate
  "Validate a user."
  [{:keys [data db]}]
  (ok (validation/validate-errors {:db db} ::validation/signup data)))
