(ns burningswell.api.authorization
  (:require [buddy.auth.accessrules :as access :refer [success error]]
            [io.pedestal.interceptor :refer [interceptor]]
            [io.pedestal.interceptor.chain :refer [terminate]]
            [buddy.auth :refer [authenticated?]]
            [burningswell.api.client :as api]
            [burningswell.db.comments :as comments]
            [burningswell.db.ratings :as ratings]
            [burningswell.db.sessions :as sessions]
            [burningswell.db.spots :as spots]
            [burningswell.db.users :as users]
            [datumbazo.core :refer [with-connection]]
            [ring.util.response :as ring]))

(def ^:private realm
  "Burning Swell API")

(def unauthorized
  {:status 401
   :headers {}
   :body
   {:errors
    [{:error :unauthorized
      :description "You must be authenticated to make this request."}]}})

(def forbidden
  {:status 403
   :headers {}
   :body
   {:errors
    [{:error :forbidden
      :description "You are not allowed to make this request."}]}})

(defn on-error
  "Authorization error handler."
  [request value]
  (cond
    (ring/response? value)
    value
    (nil? (:identity request))
    unauthorized
    :else forbidden))

(defn- db
  "Return the database from `request`."
  [request]
  (:db request))

(defn- resource-id
  "Return the resource id matching `pattern` from `request`."
  [request pattern]
  (some-> (re-matches pattern (:uri request))
          last Long/parseLong))

(defn- spot-id
  "Return the spot id from `request`."
  [request]
  (resource-id request #"/spots/(\d+)"))

(defn- comment-id
  "Return the comment id from `request`."
  [request]
  (resource-id request #"/comments/(\d+)"))

(defn- session-id
  "Return the session id from `request`."
  [request]
  (resource-id request #"/sessions/(\d+)"))

(defn- rating-id
  "Return the rating id from `request`."
  [request]
  (resource-id request #"/ratings/(\d+)"))

(defn- user-id
  "Return the user id from `request`."
  [request]
  (resource-id request #"/surfers/(\d+).*"))

(defn authenticated
  "Check that the request is autheticated."
  [request]
  (if (authenticated? request)
    true
    (error unauthorized)))

(defn- role
  "Check that the current user has `role`."
  [request role]
  (with-connection [db (db request)]
    (users/has-role-name? db (:identity request) role)))

(defn- admin
  "Check that the current user is an admin."
  [request]
  (role request :admin))

(defn- surfer
  "Check that the current user is a surfer."
  [request]
  (role request :surfer))

(defn- comment-owner
  "Check that the current user is the comment owner."
  [request]
  (with-connection [db (db request)]
    (if-let [comment (comments/by-id db (comment-id request))]
      (= (-> comment :_embedded :user :id)
         (-> request :identity :id))
      (error (ring/not-found nil)))))

(defn- rating-owner
  "Check that the current user is the rating owner."
  [request]
  (with-connection [db (db request)]
    (if-let [rating (ratings/rating-by-id db (rating-id request))]
      (= (-> rating :user-id)
         (-> request :identity :id))
      (error (ring/not-found nil)))))

(defn- spot-owner
  "Check that the current user is the spot owner."
  [request]
  (with-connection [db (db request)]
    (if-let [spot (spots/by-id db (spot-id request))]
      (= (-> spot :_embedded :user :id)
         (-> request :identity :id))
      (error (ring/not-found nil)))))

(defn- session-owner
  "Check that the current user is the session owner."
  [request]
  (with-connection [db (db request)]
    (if-let [session (sessions/session-by-id db (session-id request))]
      (= (-> session :user-id)
         (-> request :identity :id))
      (error (ring/not-found nil)))))

(defn- user-owner
  "Check that the current user is the user owner."
  [request]
  (with-connection [db (db request)]
    (if-let [user (users/by-id db (user-id request))]
      (= (-> user :id)
         (-> request :identity :id))
      (error (ring/not-found nil)))))

(def access-rules
  "The access rules for the API."
  [{:uri "/addresses"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/addresses/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}

   {:uri "/airports"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/airports/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}

   {:uri "/auth/jwt/token"
    :handler {:and [authenticated]}
    :request-method :post}

   {:uri "/comments"
    :handler authenticated
    :request-method :post}
   {:uri "/comments/:id"
    :handler {:and [authenticated {:or [admin comment-owner]}]}
    :request-method #{:delete :put}}

   {:uri "/continents"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/continents/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}

   {:uri "/countries"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/countries/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}

   {:uri "/photos"
    :handler {:and [authenticated surfer]}
    :request-method :post}

   {:uri "/photos/:id/likes"
    :handler {:and [authenticated surfer]}
    :request-method #{:delete :post}}

   {:uri "/ports"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/ports/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}

   {:uri "/ratings"
    :handler authenticated
    :request-method :post}
   {:uri "/ratings/:id"
    :handler {:and [authenticated {:or [admin rating-owner]}]}
    :request-method #{:delete :put}}

   {:uri "/regions"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/regions/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}

   {:uri "/roles"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/roles/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}

   {:uri "/sessions"
    :handler authenticated
    :request-method :post}
   {:uri "/sessions/:id"
    :handler {:and [authenticated {:or [admin session-owner]}]}
    :request-method #{:delete :put}}

   {:uri "/spots"
    :handler authenticated
    :request-method :post}
   {:uri "/spots/:id"
    :handler {:and [authenticated {:or [admin spot-owner]}]}
    :request-method #{:delete :put}}

   {:uri "/surfers/me"
    :handler {:and [authenticated]}
    :request-method #{:get}}

   {:uri "/surfers/:id"
    :handler {:and [authenticated {:or [admin user-owner]}]}
    :request-method #{:delete :put}}

   {:uri "/surfers/:id/settings"
    :handler {:and [authenticated {:or [admin user-owner]}]}
    :request-method #{:get :put}}

   {:uri "/weather/datasets"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/weather/datasets/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}

   {:uri "/weather/models"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/weather/models/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}

   {:uri "/weather/variables"
    :handler {:and [authenticated admin]}
    :request-method :post}
   {:uri "/weather/variables/:id"
    :handler {:and [authenticated admin]}
    :request-method #{:delete :put}}])

(defn authorization-interceptor
  [{:keys [policy rules] :or {policy :allow} :as opts}]
  (when (nil? rules)
    (throw (IllegalArgumentException. "rules should not be empty.")))
  (let [accessrules (access/compile-access-rules rules)]
    (interceptor
     {:name ::authorization
      :enter
      (fn [{:keys [request] :as context}]
        (if-let [match (#'access/match-access-rules accessrules request)]
          (let [res (#'access/apply-matched-access-rule match request)]
            (if (access/success? res)
              context
              (-> (terminate context)
                  (update :response #(access/handle-error
                                      % request (merge opts match))))))
          (case policy
            :allow
            context
            :reject
            (-> (terminate context)
                (update :response #(access/handle-error
                                    (error nil) request opts))))))})))
