(ns burningswell.api.routes
  (:require [burningswell.api.addresses :as addresses]
            [burningswell.api.airports :as airports]
            [burningswell.api.comments :as comments]
            [burningswell.api.config :as config]
            [burningswell.api.continents :as continents]
            [burningswell.api.countries :as countries]
            [burningswell.api.facebook :as facebook]
            [burningswell.api.geocode :as geocode]
            [burningswell.api.interceptors :as interceptors
             :refer [bounding-box location zoom]]
            [burningswell.api.jwt :as jwt]
            [burningswell.api.map :as map]
            [burningswell.api.photos :as photos]
            [burningswell.api.ports :as ports]
            [burningswell.api.ratings :as ratings]
            [burningswell.api.regions :as regions]
            [burningswell.api.roles :as roles]
            [burningswell.api.root :as root]
            [burningswell.api.search :as search]
            [burningswell.api.sessions :as sessions]
            [burningswell.api.spots :as spots]
            [burningswell.api.time-zones :as time-zones]
            [burningswell.api.users :as users]
            [burningswell.api.weather.models :as weather-models]
            [burningswell.api.weather.variables :as weather-variables]
            [io.pedestal.http.route :as routes]
            [io.pedestal.http.route.definition :refer [defroutes]]))

(def default-per-page
  "The default number of resources per page."
  12)

(def pagination
  "The default pagination interceptor."
  (interceptors/pagination default-per-page))

(defroutes routes
  [[["/" {:get [:root root/index]}]

    ["/auth/jwt/token"
     {:post [:create-jwt-token jwt/create-token]}]

    ["/airports"
     ^:interceptors [bounding-box location pagination]
     {:get [:airports airports/airports]
      :post [:create-airport airports/create-airport]}
     ["/:id"
      {:delete [:delete-airport airports/delete-airport]
       :get [:airport airports/airport]
       :put [:update-airport airports/update-airport]}]]

    ["/addresses"
     ^:interceptors [bounding-box location pagination]
     {:get [:addresses addresses/addresses]
      :post [:create-address addresses/create-address]}
     ["/:id"
      {:delete [:delete-address addresses/delete-address]
       :get [:address addresses/address]
       :put [:update-address addresses/update-address]}]]

    ["/config" {:get [:config `config/config]}]

    ["/continents"
     ^:interceptors [bounding-box location pagination]
     {:get [:continents continents/continents]
      :post [:create-continent continents/create-continent]}
     ["/:id"
      {:delete [:delete-continent continents/delete-continent]
       :get [:continent continents/continent]
       :put [:update-continent continents/update-continent]}
      ["/airports" {:get [:airports-on-continent continents/airports]}]
      ["/countries" {:get [:countries-on-continent continents/countries]}]
      ["/ports" {:get [:ports-on-continent continents/ports]}]
      ["/regions" {:get [:regions-on-continent continents/regions]}]
      ["/spots" {:get [:spots-on-continent continents/spots]}]
      ["/surfers" {:get [:users-on-continent continents/users]}]]]

    ["/countries"
     ^:interceptors [bounding-box location pagination]
     {:get [:countries countries/countries]
      :post [:create-country countries/create-country]}
     ["/:id"
      {:delete [:delete-country countries/delete-country]
       :get [:country countries/country]
       :put [:update-country countries/update-country]}
      ["/airports" {:get [:airports-in-country countries/airports]}]
      ["/image.png" {:get [:country-image countries/image]}]
      ["/ports" {:get [:ports-in-country countries/ports]}]
      ["/regions" {:get [:regions-in-country countries/regions]}]
      ["/spots" {:get [:spots-in-country countries/spots]}]
      ["/surfers" {:get [:users-in-country countries/users]}]]]

    ["/comments"
     ^:interceptors [pagination]
     {:get [:comments comments/comments]
      :post [:create-comment comments/create-comment]}
     ["/:id"
      {:delete [:delete-comment comments/delete-comment]
       :get [:comment comments/comment]
       :put [:update-comment comments/update-comment]}]]

    ["/geocode"
     ["/internet-address"
      {:get [:geocode-current-internet-address geocode/internet-address]}]
     ["/internet-address/:internet-address"
      {:get [:geocode-internet-address geocode/internet-address]}]]

    ["/map"
     ^:interceptors [bounding-box location pagination zoom]
     {:get [:map map/map]}]

    ["/oauth/facebook/login" {:get [:facebook-login facebook/login]}]

    ["/photos"
     ^:interceptors [bounding-box location pagination]
     {:get [:photos photos/photos]
      :post [:create-photo photos/create-photo]}
     ["/:id"
      {:get [:photo photos/photo]}
      ["/likes"
       {:delete [:dislike-photo photos/dislike-photo]
        :get [:photo-likes photos/photo-likes]
        :post [:like-photo photos/like-photo]}]]]

    ["/ports"
     ^:interceptors [bounding-box location pagination]
     {:get [:ports ports/ports]
      :post [:create-port ports/create-port]}
     ["/:id"
      {:delete [:delete-port ports/delete-port]
       :get [:port ports/port]
       :put [:update-port ports/update-port]}]]

    ["/ratings"
     ^:interceptors [pagination]
     {:get [:ratings ratings/ratings]
      :post [:create-rating ratings/create-rating]}
     ["/:id"
      {:delete [:delete-rating ratings/delete-rating]
       :get [:rating ratings/rating]
       :put [:update-rating ratings/update-rating]}]]

    ["/regions"
     ^:interceptors [bounding-box location pagination]
     {:get [:regions regions/regions]
      :post [:create-region regions/create-region]}
     ["/:id"
      {:delete [:delete-region regions/delete-region]
       :get [:region regions/region]
       :put [:update-region regions/update-region]}
      ["/airports" {:get [:airports-in-region regions/airports]}]
      ["/geo-json" {:get [:geo-json-region regions/geo-json]}]
      ["/image.png" {:get [:region-image regions/image]}]
      ["/ports" {:get [:ports-in-region regions/ports]}]
      ["/spots" {:get [:spots-in-region regions/spots]}]
      ["/surfers" {:get [:users-in-region regions/users]}]]]

    ["/roles"
     ^:interceptors [pagination]
     {:get [:roles roles/roles]
      :post [:create-role roles/create-role]}
     ["/:id"
      {:delete [:delete-role roles/delete-role]
       :get [:role roles/role]
       :put [:update-role roles/update-role]}]]

    ["/sessions"
     ^:interceptors [pagination]
     {:get [:sessions sessions/sessions]
      :post [:create-session sessions/create-session]}
     ["/:id"
      {:delete [:delete-session sessions/delete-session]
       :get [:session sessions/session]
       :put [:update-session sessions/update-session]}]]

    ["/search"
     ^:interceptors [bounding-box location pagination]
     ["/autocomplete" {:get [:search-autocomplete search/autocomplete]}]
     ["/details" {:get [:search-details search/details]}]]

    ["/spots"
     ^:interceptors [bounding-box location pagination]
     {:get [:spots spots/spots]
      :post [:create-spot spots/create-spot]}
     ["/:id"
      {:delete [:delete-spot spots/delete-spot]
       :get [:spot spots/spot]
       :put [:update-spot spots/update-spot]}
      ["/spots-around"
       ^:interceptors [bounding-box location pagination]
       {:get [:spots-around spots/spots-around]}]
      ["/comments" {:get [:spot-comments spots/comments]}]
      ["/photos" {:get [:spot-photos spots/photos]}]
      ["/ratings" {:get [:spot-ratings spots/ratings]}]
      ["/sessions" {:get [:spot-sessions spots/sessions]}]
      ["/weather" {:get [:spot-weather spots/weather]}]]]

    ["/time-zones"
     ^:interceptors [pagination]
     {:get [:time-zones time-zones/time-zones]}
     ["/:id" {:get [:time-zone time-zones/time-zone]}]]

    ["/surfers"
     ^:interceptors [bounding-box location pagination]
     {:get [:users users/users]
      :post [:create-user users/create-user]}
     ["/:id"
      ^:constraints {:id #"[0-9]+"}
      {:delete [:delete-user users/delete-user]
       :get [:user users/user]
       :put [:update-user users/update-user]}
      ["/settings" {:get [:user-settings users/settings]
                    :put [:save-user-settings users/save-settings]}]]
     ["/me" {:get [:me users/me]}]
     ["/validate" {:post [:validate-user users/validate]}]]

    ["/weather/models"
     ^:interceptors [pagination]
     {:get [:weather-models weather-models/models]
      :post [:create-weather-model weather-models/create-model]}
     ["/:id"
      {:delete [:delete-weather-model weather-models/delete-model]
       :get [:weather-model weather-models/model]
       :put [:update-weather-model weather-models/update-model]}]]

    ["/weather/variables"
     ^:interceptors [pagination]
     {:get [:weather-variables weather-variables/variables]
      :post [:create-weather-variable weather-variables/create-variable]}
     ["/:id"
      {:delete [:delete-weather-variable weather-variables/delete-variable]
       :get [:weather-variable weather-variables/variable]
       :put [:update-weather-variable weather-variables/update-variable]}]]]])

(def url-for
  (routes/url-for-routes routes))
