(ns burningswell.api.comments
  "The comments of users on spots."
  (:refer-clojure :exclude [comment])
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.comments :as comments]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s]))

(defn- comment-not-found
  "Return a 404 response for a comment that could not be found by `id`."
  [id]
  (not-found (format "Comment %s not found" id)))

(defapi comments
  "List all comments."
  [{:keys [api-client db query-params]}]
  (let [comments (comments/all db query-params)
        comments (hal/links api-client :comment comments)]
    (ok comments) ))

(defapi create-comment
  "Create a new comment."
  [{:keys [api-client data broker db identity path-params]}]
  (s/validate CreateComment data)
  (let [data (assoc-in data [:_embedded :user] identity)
        comment (comments/insert db data)
        comment (hal/link api-client :comment comment)]
    (publish broker "comments.created" comment)
    (created comment)))

(defapi comment
  "Show a comment."
  [{:keys [api-client db path-params]}]
  (if-let [comment (comments/by-id db (:id path-params))]
    (ok (hal/link api-client :comment comment))
    (comment-not-found (:id path-params))))

(defapi delete-comment
  "Delete a comment."
  [{:keys [broker db path-params]}]
  (if-let [comment (comments/by-id db (:id path-params))]
    (do (comments/delete db comment)
        (publish broker "comments.deleted" comment)
        (no-content))
    (comment-not-found (:id path-params))))

(defapi update-comment
  "Update a comment."
  [{:keys [api-client data broker db identity path-params]}]
  (s/validate CreateComment data)
  (if-let [comment (comments/by-id db (:id path-params))]
    (let [data (-> (assoc data :id (:id path-params))
                   (assoc-in [:_embedded :user] identity))
          comment (comments/update db data)
          comment (hal/link api-client :comment comment)]
      (publish broker "comments.updated" comment)
      (ok comment))
    (comment-not-found (:id path-params))))
