(ns burningswell.api.regions
  "The regions of the world."
  (:require [burningswell.api.coerce :as coerce]
            [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.photos :refer [make-photo]]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.spots :refer [enhance-spots]]
            [burningswell.db.airports :as airports]
            [burningswell.db.photos :as photos]
            [burningswell.db.ports :as ports]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [burningswell.db.users :as users]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s])
  (:import java.io.ByteArrayInputStream))

(set! *warn-on-reflection* true)

(defn- region-not-found
  "Return a 404 response for a region that could not be found by `id`."
  [id]
  (not-found (format "Region %s not found" id)))

(defn geo-json
  "Return a region as GeoJSON."
  [{:keys [api-client db path-params]}]
  (if-let [region (regions/by-id db (:id path-params))]
    (ok (regions/as-geo-json db region)
        {"Content-Type" "application/vnd.geo+json"})
    (region-not-found (:id path-params))))

(defapi regions
  "List all regions."
  [{:keys [api-client db query-params]}]
  (let [regions (regions/all db query-params)
        regions (regions/assoc-photo db regions)]
    (ok (hal/links api-client :region regions))))

(defapi create-region
  "Create a new region."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateRegion data)
  (let [region (regions/insert db data)
        region (hal/link api-client :region region)]
    (publish broker "regions.created" region)
    (created region)))

(defapi region
  "Show a region."
  [{:keys [api-client db path-params]}]
  (if-let [region (regions/by-id db (:id path-params))]
    (ok (hal/link api-client :region region))
    (region-not-found (:id path-params))))

(defapi delete-region
  "Delete a region."
  [{:keys [broker db path-params]}]
  (if-let [region (regions/by-id db (:id path-params))]
    (do (regions/delete db region)
        (publish broker "regions.deleted" region)
        (no-content))
    (region-not-found (:id path-params))))

(defapi update-region
  "Update a region."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateRegion data)
  (if-let [region (regions/by-id db (:id path-params))]
    (let [region (regions/update db (merge region data))
          region (hal/link api-client :region region)]
      (publish broker "regions.updated" region)
      (ok region))
    (region-not-found (:id path-params))))

(defapi airports
  "List all airports in a region."
  [{:keys [api-client data broker db path-params query-params]}]
  (if-let [region (regions/by-id db (:id path-params))]
    (let [airports (airports/in-region db region query-params)]
      (ok (hal/links api-client :airport airports)))
    (region-not-found (:id path-params))))

(defapi image
  "Return the region image."
  [{:keys [db path-params query-params]}]
  (if-let [region (regions/by-id db (:id path-params))]
    (let [params (coerce/update-color query-params)]
      (ok (ByteArrayInputStream.
           (regions/as-png db region params))
          {"cache-control" "public, max-age=86400"
           "content-type" "image/png"}))
    (region-not-found (:id path-params))))

(defapi photos
  "List all photos of the region."
  [{:keys [api-client db path-params query-params]}]
  (when-let [region (regions/by-id db (:id path-params))]
    (->> (photos/by-region db region query-params)
         (photos/assoc-images db)
         (map #(make-photo api-client %))
         (ok))))

(defapi ports
  "List all ports in a region."
  [{:keys [api-client data broker db path-params query-params]}]
  (if-let [region (regions/by-id db (:id path-params))]
    (let [ports (ports/in-region db region query-params)]
      (ok (hal/links api-client :port ports)))
    (region-not-found (:id path-params))))

(defapi spots
  "List all spots in a region."
  [{:keys [api-client data broker db path-params query-params]}]
  (if-let [region (regions/by-id db (:id path-params))]
    (let [spots (spots/in-region db region query-params)
          spots (enhance-spots api-client db spots query-params)]
      (ok spots))
    (region-not-found (:id path-params))))

(defapi users
  "List all users in a region."
  [{:keys [api-client data broker db path-params query-params]}]
  (if-let [region (regions/by-id db (:id path-params))]
    (->> (users/in-region db region query-params)
         (map #(users/safe-user identity %))
         (hal/links api-client :user)
         (ok))
    (region-not-found (:id path-params))))

(set! *warn-on-reflection* false)
