(ns burningswell.api.spec.core
  (:require [burningswell.api.spec.context :refer [*context*]]
            [burningswell.api.spec.generators :as generators]
            [clojure.spec :as s]
            [clojure.spec.gen :as gen]
            [clojure.string :as str]
            [burningswell.db.countries :as countries]
            [burningswell.db.users :as users]
            [burningswell.db.spots :as spots]
            [burningswell.db.regions :as regions]
            [burningswell.db.specs :as specs]
            [geo.core :refer [point?]]
            [geo.postgis :refer [point]]))

(def email-regex
  "The regular expression for an email address."
  #"^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,63}$")

(defn max-length [len]
  #(<= (count %) len))

(defn min-length [len]
  #(>= (count %) len))

(defn not-blank? [s]
  (not (str/blank? s)))

(defn spot-id?
  "Returns true if `id` is an existing country id, otherwise false."
  [id]
  (spots/exists? (:db *context*) {:id id}))

(defn country-id?
  "Returns true if `id` is an existing country id, otherwise false."
  [id]
  (countries/exists? (:db *context*) {:id id}))

(defn region-id?
  "Returns true if `id` is an existing region id, otherwise false."
  [id]
  (regions/exists? (:db *context*) {:id id}))

(defn email-available?
  "Returns true if `email` is available, otherwise false."
  [email]
  (users/email-available? (:db *context*) email))

(defn username-available?
  "Returns true if `username` is available, otherwise false."
  [username]
  (users/username-available? (:db *context*) username))

(defn email?
  "Returns true if `s` is a valid email address, otherwise false."
  [s]
  (re-matches email-regex s))

(defn units?
  "Returns true if `x` is a valid unit system, otherwise false."
  [x]
  (contains? #{"eu" "uk" "us"} x))

(s/def ::country-id
  (s/with-gen (s/and pos-int? country-id?)
    (constantly generators/country-id)))

(s/def ::region-id
  (s/with-gen (s/and pos-int? region-id?)
    (constantly generators/region-id)))

(s/def ::spot-id
  (s/with-gen (s/and pos-int? spot-id?)
    (constantly generators/spot-id)))

(s/def ::latitude
  (s/double-in :min -90 :max 90 :NaN? false :infinite? false))

(s/def ::longitude
  (s/double-in :min -180 :max 180 :NaN? false :infinite? false))

(s/def ::location
  (s/with-gen (s/spec point?)
    #(gen/fmap (fn [[x y]] (point 4326 x y))
               (gen/tuple (s/gen ::longitude)
                          (s/gen ::latitude)))))
