(ns burningswell.api.continents
  "The continents of the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.spots :refer [enhance-spots]]
            [burningswell.db.airports :as airports]
            [burningswell.db.continents :as continents]
            [burningswell.db.countries :as countries]
            [burningswell.db.ports :as ports]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [burningswell.db.users :as users]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn- continent-not-found
  "Return a 404 response for a continent that could not be found by `id`."
  [id]
  (not-found (format "Continent %s not found" id)))

(defapi continent
  "Return a continent by id."
  [{:keys [api-client db path-params]}]
  (if-let [continent (continents/by-id db (:id path-params))]
    (ok (hal/link api-client :continent continent))
    (continent-not-found (:id path-params))))

(defapi continents
  "List all continents."
  [{:keys [api-client db params]}]
  (let [continents (continents/all db params)]
    (ok (hal/links api-client :continent continents))))

(defapi create-continent
  "Create a new continent."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateContinent data)
  (let [continent (continents/insert db data)]
    (publish broker "continents.created" continent)
    (created (hal/link api-client :continent continent))))

(defapi delete-continent
  "Delete a continent."
  [{:keys [broker db path-params]}]
  (if-let [continent (continents/by-id db (:id path-params))]
    (do (continents/delete db continent)
        (publish broker "continents.deleted" continent)
        (no-content))
    (continent-not-found (:id path-params))))

(defapi update-continent
  "Update a continent."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateContinent data)
  (if-let [continent (continents/by-id db (:id path-params))]
    (let [continent (continents/update db (merge continent data))
          continent (hal/link api-client :continent continent)]
      (publish broker "continents.updated" continent)
      (ok continent))
    (continent-not-found (:id path-params))))

(defapi airports
  "List all airports on a continent."
  [{:keys [api-client db path-params query-params]}]
  (if-let [continent (continents/by-id db (:id path-params))]
    (let [airports (airports/in-continent db continent query-params)]
      (ok (hal/links api-client :airport airports)))
    (continent-not-found (:id path-params))))

(defapi countries
  "List all countries on a continent."
  [{:keys [api-client db path-params query-params]}]
  (if-let [continent (continents/by-id db (:id path-params))]
    (let [countries (countries/in-continent db continent query-params)]
      (ok (hal/links api-client :country countries)))
    (continent-not-found (:id path-params))))

(defapi ports
  "List all ports on a continent."
  [{:keys [api-client db path-params query-params]}]
  (if-let [continent (continents/by-id db (:id path-params))]
    (let [ports (ports/in-continent db continent query-params)]
      (ok (hal/links api-client :port ports)))
    (continent-not-found (:id path-params))))

(defapi regions
  "List all regions on a continent."
  [{:keys [api-client db path-params query-params]}]
  (if-let [continent (continents/by-id db (:id path-params))]
    (let [regions (regions/in-continent db continent query-params)]
      (ok (hal/links api-client :region regions)))
    (continent-not-found (:id path-params))))

(defapi spots
  "List all spots on a continent."
  [{:keys [api-client db path-params query-params]}]
  (if-let [continent (continents/by-id db (:id path-params))]
    (let [spots (spots/in-continent db continent query-params)
          spots (enhance-spots api-client db spots query-params)]
      (ok spots))
    (continent-not-found (:id path-params))))

(defapi users
  "List all users on a continent."
  [{:keys [api-client db path-params query-params]}]
  (if-let [continent (continents/by-id db (:id path-params))]
    (->> (users/in-continent db continent query-params)
         (map #(users/safe-user identity %))
         (hal/links api-client :user)
         (ok))
    (continent-not-found (:id path-params))))

(set! *warn-on-reflection* false)
