(ns burningswell.api.errors
  (:require [burningswell.api.i18n :refer [t]]
            [burningswell.api.spec.user :as user]
            [burningswell.api.spec.spot :as spot]
            [clojure.spec :as s]
            [clojure.string :as str]))

(defn i18n-key
  "Return the I18n attribute key for `problem`."
  [problem]
  (-> problem :via last))

(defn set-as-sentence
  "Return the members of `s` as a sentence."
  [s]
  (let [s (sort s)
        size (count s)
        escape #(str \" % \")
        separator (str " " (t :en :or) " ")]
    (cond
      (= size 1)
      (escape (first s))
      (= size 2)
      (str/join separator (map escape s))
      (> size 2)
      (str (str/join ", " (map escape (butlast s)))
           separator (escape (last s))))))

(defn dispatch-key
  "Return the dispatch key for `problem`."
  [{:keys [pred via] :as problem}]
  (conj via pred))

(defn- pprint-problem [problem]
  (println "DISPATCH: "  (dispatch-key problem))
  (clojure.pprint/pprint problem)
  (println "\n"))

(defn error-msg-predicate
  [{:keys [pred via] :as problem}]
  (cond
    (set? pred)
    'not-a-set-member
    (symbol? pred)
    pred
    (and (sequential? pred)
         (symbol? (first pred)))
    (first pred)))

(defmulti error-path
  "Adds a :key-path to a problem."
  dispatch-key)

(defmethod error-path 'contains?
  [{:keys [path pred]}]
  (conj path (last pred)))

(defmethod error-path [::spot/create-spot 'location-available?] [problem]
  [:location])

(defmethod error-path [::spot/update-spot 'location-available?] [problem]
  [:location])

(defmethod error-path :default
  [{:keys [path] :as problem}]
  (if (empty? path)
    [(-> problem :via last name keyword)]
    path))

(defmethod error-path :default
  [{:keys [in path] :as problem}]
  (cond
    (not (empty? in)) in
    (not (empty? path)) path
    :else
    (let [predicate (error-msg-predicate problem)]
      (if-let [method (get (methods error-path) predicate)]
        (method problem)))))

(defmethod error-path :default
  [{:keys [in path via] :as problem}]
  (let [predicate (error-msg-predicate problem)]
    (if-let [method (get (methods error-path) predicate)]
      (method problem)
      (cond
        (not (empty? in)) in
        (not (empty? path)) path
        :else (map #(-> % name keyword) via)))))

;; Error messages

(defmulti error-msg
  "Returns a human readable string of a problem."
  dispatch-key)

(defmethod error-msg 'contains? [problem]
  (str (t :en (-> problem :pred last)) " "
       (t :en :contains?) "."))

(defmethod error-msg 'not-a-set-member [problem]
  (str (t :en (i18n-key problem)) " "
       (t :en :must-be) " "
       (set-as-sentence (:pred problem))
       "."))

(defmethod error-msg 'email? [problem]
  (str (t :en :invalid-email) "."))

(defmethod error-msg 'email-available? [problem]
  (str (t :en :email-available?) "."))

(defmethod error-msg 'max-length [problem]
  (let [[_ length] (:pred problem)]
    (str (t :en (i18n-key problem)) " "
         (t :en :max-length length) ".")))

(defmethod error-msg 'min-length [problem]
  (let [[_ length] (:pred problem)]
    (str (t :en (i18n-key problem)) " "
         (t :en :min-length length) ".")))

(defmethod error-msg 'not-blank? [problem]
  (str (t :en (i18n-key problem)) " "
       (t :en :not-blank?) "."))

(defmethod error-msg 'string? [problem]
  (str (t :en (i18n-key problem)) " "
       (t :en :not-string) "."))

(defmethod error-msg 'username-available? [problem]
  (str (t :en :username-available?) "."))

(defmethod error-msg 'units? [problem]
  (str (t :en :units?) "."))

(defmethod error-msg 'point? [problem]
  (str (t :en (i18n-key problem)) " "
       (t :en :point?) "."))

(defmethod error-msg 'pos-int? [problem]
  (str (t :en (i18n-key problem)) " "
       (t :en :pos-int?) "."))

(defmethod error-msg [::spot/create-spot 'location-available?] [problem]
  (str (t :en ::spot/location-taken) "."))

(defmethod error-msg [::spot/update-spot 'location-available?] [problem]
  (str (t :en ::spot/location-taken) "."))

(defmethod error-msg :default [problem]
  (let [predicate (error-msg-predicate problem)]
    (if-let [method (get (methods error-msg) predicate)]
      (method problem)
      (do (pprint-problem problem)
          (str (t :en :unknown-error) ".")))))

(defn add-error
  "Add the human readable error to the `problem`."
  [problem]
  ;; (pprint-problem problem)
  (->> {:message (error-msg problem)
        :path (error-path problem)}
       (assoc problem :error)))

(defn add-errors
  "Add human readable errors to the `problems`."
  [problems]
  (update problems ::s/problems #(map add-error %)))

(defn errors
  "Extract only the errors from `problems`."
  [spec problems]
  (reduce
   (fn [errors problem]
     (let [{:keys [message path]} (:error problem)]
       (update-in errors path #(conj (or % []) message))))
   {} (::s/problems problems)))

(defn explain-data
  [spec data]
  (-> (s/explain-data spec data)
      (add-errors)))

(defn explain-errors
  [spec data]
  (->> (explain-data spec data)
       (errors spec )
       (not-empty)))
