(ns burningswell.api.ports
  "The ports in the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.ports :as ports]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn port-not-found
  "Return a 404 response for a port that could not be found by `id`."
  [id]
  (not-found (format "Port %s not found" id)))

(defapi ports
  "List all ports."
  [{:keys [api-client db query-params]}]
  (let [ports (ports/all db query-params)]
    (ok (hal/links api-client :port ports))))

(defapi create-port
  "Create a new port."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreatePort data)
  (let [port (ports/insert db data)
        port (hal/link api-client :port port)]
    (publish broker "ports.created" port)
    (created port)))

(defapi port
  "Show a port."
  [{:keys [api-client db path-params]}]
  (if-let [port (ports/by-id db (:id path-params))]
    (ok (hal/link api-client :port port))
    (port-not-found (:id path-params))))

(defapi delete-port
  "Delete a port."
  [{:keys [broker db path-params]}]
  (if-let [port (ports/by-id db (:id path-params))]
    (do (ports/delete db port)
        (publish broker "ports.deleted" port)
        (no-content))
    (port-not-found (:id path-params))))

(defapi update-port
  "Update a port."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreatePort data)
  (if-let [port (ports/by-id db (:id path-params))]
    (let [port (ports/update db (merge port data))
          port (hal/link api-client :port port)]
      (publish broker "ports.updated" port)
      (ok port))
    (port-not-found (:id path-params))))

(set! *warn-on-reflection* false)
