(ns burningswell.api.roles
  "The roles of users."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.roles :as roles]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn role-not-found
  "Return a 404 response for a role that could not be found by `id`."
  [id]
  (not-found (format "Role %s not found" id)))

(defapi roles
  "List all roles."
  [{:keys [api-client db query-params]}]
  (let [roles (roles/all db query-params)]
    (ok roles)))

(defapi create-role
  "Create a new role."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateRole data)
  (let [role (roles/insert db data)]
    (publish broker "roles.created" role)
    (created role)))

(defapi role
  "Show a role."
  [{:keys [api-client db path-params]}]
  (if-let [role (roles/by-id db (:id path-params))]
    (ok role)
    (role-not-found (:id path-params))))

(defapi delete-role
  "Delete a role."
  [{:keys [broker db path-params]}]
  (if-let [role (roles/by-id db (:id path-params))]
    (do (roles/delete db role)
        (publish broker "roles.deleted" role)
        (no-content))
    (role-not-found (:id path-params))))

(defapi update-role
  "Update a role."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateRole data)
  (if-let [role (roles/by-id db (:id path-params))]
    (let [role (roles/update db (merge role data))]
      (publish broker "roles.updated" role)
      (ok role))
    (role-not-found (:id path-params))))

(set! *warn-on-reflection* false)
