(ns burningswell.api.schemas
  "Schemas and constructors"
  (:require [burningswell.db.schemas :as db]
            [burningswell.db.users :as users]
            [clojure.string :as str]
            [clojure.set :refer [subset?]]
            [schema.core :as s]
            [schema.spec.core :refer [precondition]]
            [schema.spec.leaf :refer [leaf-spec]]
            [schema.macros :as macros]
            [schema.utils :as utils])
  (:import [java.awt Color]
           [schema.utils NamedError ValidationError]
           [org.postgis PGbox2d Point]))

(defn explain-errors
  "Return a serializable explanation of `errors`."
  [errors]
  (cond
    (map? errors)
    (->> errors
         (map (fn [[k v]]
                [k (explain-errors v)]))
         (into {}))

    (or (seq? errors)
        (coll? errors))
    (map explain-errors errors)

    (instance? NamedError errors)
    [(.name errors)
     (utils/named-error-explain errors)]

    (instance? ValidationError errors)
    (utils/validation-error-explain errors)

    :else
    errors))

(defn check
  "Check `x` against `schema` and explain the errors."
  [schema x]
  (some-> (s/check schema x) explain-errors))

(def email-regex
  (re-pattern
   (str "^[_A-Za-z0-9-\\+]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9-]+"
        "(\\.[A-Za-z0-9]+)*(\\.[A-Za-z]{2,})$")))

(defn email?
  "Returns true if email is valid, otherwise false."
  [email]
  (re-matches email-regex email))

(s/defschema Email
  "The schema for an email address."
  (s/both s/Str (s/pred email? 'email?)))

(defrecord EmailAvailable [db]
  s/Schema
  (spec [this]
    (leaf-spec
     (fn [email]
       (if (users/email-available? db email)
         email
         (macros/validation-error this email (list 'email-available? email))))))
  (explain [this] 'email-available?))

(defn email-available? [db]
  (EmailAvailable. db))

(defrecord UsernameAvailable [db]
  s/Schema
  (spec [this]
    (leaf-spec
     (fn [username]
       (if (users/username-available? db username)
         username
         (macros/validation-error
          this username (list 'username-available? username))))))
  (explain [this] 'username-available?))

(defn username-available? [db]
  (UsernameAvailable. db))

(defn min-size? [size x]
  (>= (count x) size))

(defrecord MinSize [size]
  s/Schema
  (spec [this]
    (leaf-spec
     (precondition
      this #(min-size? size %)
      #(list 'min-size? size (utils/value-name %)))))
  (explain [this] (list 'min-size? size)))

(defn min-size [size]
  (MinSize. size))

(defn max-size? [size x]
  (<= (count x) size))

(defrecord MaxSize [size]
  s/Schema
  (spec [this]
    (leaf-spec
     (precondition
      this #(max-size? size %)
      #(list 'max-size? size (utils/value-name %)))))
  (explain [this] (list 'max-size? size)))

(defn max-size [size]
  (MaxSize. size))

(defn add-created-at
  "Assoc the :created-at timestamp to `schema`."
  [schema]
  (assoc schema :created-at s/Inst))

(defn add-updated-at
  "Assoc the :updated-at timestamp to `schema`."
  [schema]
  (assoc schema :updated-at s/Inst))

(defn add-timestamps
  "Assoc the :created-at and :updated-at timestamps to `schema`."
  [schema]
  (-> (add-created-at schema)
      (add-updated-at)))

(defn add-hal-embedded
  "Assoc the :embedded HAL resources to `schema`."
  [schema]
  (assoc schema (s/optional-key :_embedded) s/Any))

(defn add-hal-links
  "Assoc the HAL :links to `schema`."
  [schema]
  (assoc schema (s/optional-key :_links) s/Any))

(s/defschema Config
  "Schema for the configuration."
  {:api
   {:scheme s/Keyword
    :server-name String
    :server-port s/Int}
   :facebook
   {:client-id String
    :redirect-uri String}
   :flickr {:api-key String}
   :google {:maps {:api-key String}}
   :web
   {:scheme s/Keyword
    :server-name String
    :server-port s/Int}})

(s/defschema Units
  "Schema for weather units."
  (s/enum "eu" "uk" "us"))

(s/defschema Link
  "Schema for a HAL link."
  {:href s/Str})

(s/defschema Id
  "Schema for an identifer."
  s/Int)

(s/defschema Location
  "Schema for a location."
  s/Any)

(defn- add-links [schema & links]
  (reduce (fn [schema link]
            (assoc-in schema [:_links link] Link))
          schema links))

(s/defschema EmbeddedContinent
  "Schema for an embedded continent."
  (add-links db/EmbeddedContinent :self))

(s/defschema EmbeddedCountry
  "Schema for an embedded country."
  (add-links db/EmbeddedCountry :self))

(s/defschema EmbeddedUser
  "Schema for an embedded user."
  (add-links db/EmbeddedUser :self))

(s/defschema EmbeddedSpot
  "Schema for an embedded spot."
  (add-links db/EmbeddedSpot :self))

;; Time Zone

(s/defschema TimeZone
  "Schema for a time zone."
  (-> db/TimeZone
      (add-hal-links)))

(s/defschema EmbeddedTimeZone
  "Schema for an embedded time zone."
  (-> db/EmbeddedTimeZone
      (add-hal-links)))

(s/defschema EmbeddedPhoto
  "Schema for an embedded photo."
  (-> db/EmbeddedPhoto
      (dissoc :s3-key)
      (add-links :self)))

;; Photos

(s/defschema Photo
  "Schema for a photo."
  (-> db/Photo
      (assoc
       :_embedded
       {:user (s/maybe EmbeddedUser)}
       :_links
       {:self Link
        :likes Link
        (s/optional-key :spot) Link
        (s/optional-key :user) Link})
      (dissoc :s3-key :url)))

;; Continents

(s/defschema Continent
  "Schema for a continent."
  (-> db/Continent
      (add-hal-links)))

;; Countries

(s/defschema CountryEmbedded
  "Schema for embedded HAL country resources."
  {:continent EmbeddedContinent
   (s/optional-key :photo) (s/maybe EmbeddedPhoto)})

(s/defschema CountryLinks
  "Schema for HAL country links."
  {:airports Link
   :continent Link
   :ports Link
   :regions Link
   :self Link
   :spots Link
   :users Link
   (s/optional-key :photo) Link})

(s/defschema Country
  "Schema for a country."
  (assoc db/Country
         :_embedded CountryEmbedded
         :_links CountryLinks))

;; Regions

(s/defschema EmbeddedRegion
  "Schema for an embedded region."
  (add-links db/EmbeddedRegion :self))

(s/defschema RegionEmbedded
  "Schema for embedded HAL resources in a region."
  {:country EmbeddedCountry
   (s/optional-key :photo) (s/maybe EmbeddedPhoto)})

(s/defschema RegionLinks
  "Schema for HAL region links."
  {:airports Link
   :country Link
   :ports Link
   :self Link
   :spots Link
   :users Link
   (s/optional-key :photo) Link})

(s/defschema Region
  "Schema for a region."
  (assoc db/Region
         :_embedded RegionEmbedded
         :_links RegionLinks))

;; Addresses

(s/defschema Address
  "Schema for an address."
  (assoc db/Address
         :_embedded
         {:country EmbeddedCountry
          :region (s/maybe EmbeddedRegion)
          :user (s/maybe EmbeddedUser)}
         :_links
         {:country Link
          :self Link
          (s/optional-key :region) Link
          (s/optional-key :user) Link}))

(s/defschema CreateAddress
  "Schema to create an address."
  {:_embedded
   {:country {:id (s/maybe s/Int)}
    :region {:id (s/maybe s/Int)}}
   :location Location
   (s/optional-key :city) s/Str
   (s/optional-key :country-id) s/Int
   (s/optional-key :formatted) s/Str
   (s/optional-key :postal-code) s/Str
   (s/optional-key :region-id) s/Int
   (s/optional-key :street-name) s/Str
   (s/optional-key :street-number) s/Str})

;; Airports

(s/defschema CreateAirport
  "Schema to create an airport."
  {:_embedded
   {:country {:id (s/maybe s/Int)}
    :region {:id (s/maybe s/Int)}}
   :gps-code s/Str
   :iata-code s/Str
   :location Location
   :name s/Str
   :wikipedia-url s/Str})

(s/defschema Airport
  "Schema for a airport."
  (assoc db/Airport
         :_embedded
         {:country EmbeddedCountry
          :region (s/maybe EmbeddedRegion)}
         :_links
         {:country Link
          :self Link
          (s/optional-key :region) Link}))

;; Roles

(s/defschema Role
  "Schema for a role."
  (-> {:description s/Str
       :name s/Str
       :id s/Int}
      (add-hal-links)
      (add-timestamps)))

(s/defschema PhotoLike
  "Schema for a photo."
  {:created-at s/Inst
   :like s/Bool
   :photo-id s/Int
   :updated-at s/Inst
   :user-id s/Int})

(s/defschema CreatePhoto
  "Schema to create an photo."
  {:location Location
   :title s/Str
   :url s/Str
   (s/optional-key :flickr-id) (s/maybe s/Str)})

;; Spots

(s/defschema SpotEmbedded
  "Schema for embedded HAL resources in a spot."
  {:country (s/maybe EmbeddedCountry)
   :region (s/maybe EmbeddedRegion)
   :user (s/maybe EmbeddedUser)
   :photo (s/maybe Photo)
   :time-zone (s/maybe EmbeddedTimeZone)
   (s/optional-key :weather) (s/maybe s/Any)})

(s/defschema SpotLinks
  "Schema for HAL spot links."
  {:photos Link
   :self Link
   :spots-around Link
   :weather Link
   (s/optional-key :country) Link
   (s/optional-key :user) Link
   (s/optional-key :photo) Link
   (s/optional-key :region) Link
   (s/optional-key :time-zone) Link})

(s/defschema Spot
  "Schema for a spot."
  (assoc db/Spot
         :_embedded SpotEmbedded
         :_links SpotLinks))

;; Comments

(s/defschema Comment
  "Schema for a spot."
  (assoc db/Comment
         :_embedded
         {:spot EmbeddedSpot
          :user EmbeddedUser}
         :_links
         {:self Link
          :spot Link
          :user Link}))

(s/defschema CreateComment
  "Schema to create a comment."
  {:_embedded
   {:spot {:id s/Int}}
   :content s/Str
   :visible s/Str})

;; Users

(s/defschema User
  "Schema for a user"
  (assoc db/User
         :_embedded {:country (s/maybe EmbeddedCountry)
                     :region (s/maybe EmbeddedRegion)}
         :_links
         {:self Link
          (s/optional-key :country) Link
          (s/optional-key :region) Link}))

;; Ports

(s/defschema Port
  "Schema for a port."
  (assoc db/Port
         :_embedded
         {:country EmbeddedCountry
          :region (s/maybe EmbeddedRegion)}
         :_links
         {:country Link
          :self Link
          (s/optional-key :region) Link}))

(s/defschema CreatePort
  "Schema to create an port."
  {:_embedded
   {:country {:id (s/maybe s/Int)}
    :region {:id (s/maybe s/Int)}}
   :location Location
   :name s/Str
   :type s/Str
   :website-url (s/maybe s/Str)})

(s/defschema Rating
  "Schema for a rating."
  (-> {:id s/Int
       :rating s/Int
       :spot-id s/Int
       :user-id s/Int
       :rated-at s/Inst}
      ;; TODO: Add HAL links
      ;; (add-hal-links)
      (add-hal-embedded)
      (add-timestamps)))

(s/defschema Session
  "Schema for a session."
  (-> {:id s/Int
       :spot-id s/Int
       :user-id s/Int
       (s/optional-key :started-at) s/Inst
       (s/optional-key :stopped-at) s/Inst
       :rating s/Int}
      (add-hal-embedded)
      (add-timestamps)))

(s/defschema WeatherEntry
  "Schema for the weather at a surf spot."
  {:unit s/Str
   :value (s/maybe s/Num)})

(s/defschema PrimaryWaveDirection
  "Schema for the primary wave direction."
  {(s/optional-key :dirpwsfc) WeatherEntry})

(s/defschema SecondaryWaveDirection
  "Schema for the secondary wave direction."
  {(s/optional-key :dirswsfc) WeatherEntry})

(s/defschema SignificantWaveHeight
  "Schema for the significant wave height."
  {(s/optional-key :htsgwsfc) WeatherEntry})

(s/defschema PrimaryWaveMeanPeriod
  "Schema for the primary wave mean period"
  {(s/optional-key :perpwsfc) WeatherEntry})

(s/defschema SecondaryWaveMeanPeriod
  "Schema for the secondary wave mean period."
  {(s/optional-key :perswsfc) WeatherEntry})

(s/defschema TotalCloudCover
  "Schema for the total cloud cover."
  {(s/optional-key :tcdcclm) WeatherEntry})

(s/defschema SurfaceTemperature
  "Schema for the surface temperature."
  {(s/optional-key :tmpsfc) WeatherEntry})

(s/defschema WindUCmponent
  "Schema for the U wind component."
  {(s/optional-key :ugrdsfc) WeatherEntry})

(s/defschema WindVComponent
  "Schema for the V wind component."
  {(s/optional-key :vgrdsfc) WeatherEntry})

(s/defschema WindDirection
  "Schema for the wind direction."
  {(s/optional-key :wdirsfc) WeatherEntry})

(s/defschema WindSpeed
  "Schema for the wind speed."
  {(s/optional-key :windsfc) WeatherEntry})

(s/defschema WindWaveDirection
  "Schema for the wind wave direction."
  {(s/optional-key :wvdirsfc) WeatherEntry})

(s/defschema WindWaveMeanPeriod
  "Schema for mean period of wind waves."
  {(s/optional-key :wvpersfc) WeatherEntry})

(s/defschema Weather
  "Schema for the weather at a surf spot."
  {s/Inst
   (merge PrimaryWaveDirection
          PrimaryWaveMeanPeriod
          SecondaryWaveDirection
          SecondaryWaveMeanPeriod
          SignificantWaveHeight
          SurfaceTemperature
          TotalCloudCover
          WindDirection
          WindSpeed
          WindUCmponent
          WindVComponent
          WindWaveDirection
          WindWaveMeanPeriod)})

(s/defschema WeatherDataset
  "Schema for a weather dataset."
  (-> {:das s/Str
       :dds s/Str
       :dods s/Str
       :download-started-at s/Inst
       :download-finished-at s/Inst
       :filename (s/maybe s/Str)
       :filesize (s/maybe s/Int)
       :id s/Int
       :model-id s/Int
       :reference-time s/Inst
       :valid-time s/Inst
       :variable-id s/Int}
      (add-hal-links)
      (add-hal-embedded)
      (add-timestamps)))

(s/defschema WeatherModel
  "Schema for a weather model."
  (-> db/WeatherModel
      (add-hal-links)
      (add-hal-embedded)))

(s/defschema WeatherVariable
  "Schema for a weather variable."
  (-> {:id s/Int
       :name s/Str
       :description s/Str
       :unit s/Str}
      (add-hal-links)
      (add-hal-embedded)
      (add-timestamps)))

(s/defschema CreateContinent
  "Schema to create a continent."
  {:code s/Str
   :name s/Str})

(s/defschema CreateCountry
  "Schema to create a country."
  {:_embedded
   {:continent {:id s/Int}
    (s/optional-key :photo) {:id s/Int s/Any s/Any}}
   :area s/Int
   :fips-code s/Str
   :iso-3166-1-alpha-2 s/Str
   :iso-3166-1-alpha-3 s/Str
   :iso-3166-1-numeric s/Int
   :name s/Str
   :phone-prefix s/Str
   :population s/Int})

(s/defschema CreateRating
  "Schema to create a rating."
  {:spot-id s/Int
   :rating s/Int})

(s/defschema CreateRegion
  "Schema to create a region."
  {:_embedded
   {:country {:id s/Int}
    (s/optional-key :photo) {:id s/Int s/Any s/Any}}
   :name s/Str
   (s/optional-key :location) Location})

(s/defschema CreateRole
  "Schema to create a role."
  {:name s/Str
   :description s/Str})

(defn CreateUser [db]
  {:email (s/both Email (email-available? db))
   :password (s/both s/Str (min-size 6) (max-size 64))
   :username (s/both s/Str (username-available? db) (min-size 2) (max-size 16))
   (s/optional-key :terms-of-service) (s/both s/Bool (s/pred true? 'true?))})

(def Credentials
  {:login s/Str
   :password s/Str})

(s/defschema CreateSession
  "Schema to create a session."
  {:spot-id s/Int
   :rating s/Int
   (s/optional-key :started-at) s/Inst
   (s/optional-key :stopped-at) s/Inst})

(s/defschema CreateSpot
  "Schema to create a spot."
  {(s/optional-key :_embedded)
   {(s/optional-key :country) {(s/optional-key :id) s/Int}
    (s/optional-key :region) {(s/optional-key :id) s/Int}
    (s/optional-key :photo) {(s/optional-key :id) s/Int}
    (s/optional-key :time-zone) {(s/optional-key :id) s/Int}}
   :location Location
   :name (s/both s/Str (min-size 2) (max-size 64))
   (s/optional-key :visible) s/Str})

(s/defschema CreateWeatherDataset
  "Create a weather dataset."
  {:das s/Str
   :dds s/Str
   :dods s/Str
   :download-started-at s/Inst
   :download-finished-at s/Inst
   :filename (s/maybe s/Str)
   :filesize (s/maybe s/Int)
   :model-id s/Int
   :reference-time s/Inst
   :valid-time s/Inst
   :variable-id s/Int})

(s/defschema CreateWeatherModel
  "Create a weather model."
  {:name s/Str
   :description s/Str
   :latest-reference-time (s/maybe s/Inst)
   :dods s/Str
   :res-x s/Num
   :res-y s/Num})

(s/defschema CreateWeatherVariable
  "Create a weather variable."
  {:name s/Str
   :description s/Str
   :unit s/Str})

(s/defschema PaginationParams
  "Pagination query params"
  {(s/optional-key :page) s/Int
   (s/optional-key :per-page) s/Int})

(s/defschema LocationParams
  "Location query params"
  {(s/optional-key :location) Point
   (s/optional-key :latitude) Double
   (s/optional-key :longitude) Double})

(s/defschema CountriesParams
  "Query params for countries."
  (merge LocationParams
         PaginationParams
         {(s/optional-key :min-spots) s/Int}))

(s/defschema RegionsParams
  "Query params for regions."
  (merge LocationParams
         PaginationParams
         {(s/optional-key :min-spots) s/Int}))

(s/defschema SpotParams
  "Query params for a spot."
  {(s/optional-key :units) Units})

(s/defschema BoundingBoxParams
  "The schema for bounding box query parameters."
  {:top Double
   :right Double
   :bottom Double
   :left Double
   (s/optional-key :bounding-box) PGbox2d})

(defn- bounding-box-params?
  "Return true if `m` contains bounding box params."
  [m]
  (subset? #{:top :left :bottom :right} (set (keys m))))

(defn- location-params?
  "Return true if `m` contains bounding box params."
  [m]
  (subset? #{:latitude :longitude} (set (keys m))))

(s/defschema ImageParams
  "Query params for an image."
  {(s/optional-key :color) Color
   (s/optional-key :width) Long
   (s/optional-key :height) Long})

(s/defschema SpotsParams
  "Query params for spots."
  (let [Optional
        (merge
         PaginationParams
         {(s/optional-key :units) Units
          (s/optional-key :query) s/Str})]
    (s/conditional
     bounding-box-params?
     (merge BoundingBoxParams Optional)
     location-params?
     (merge LocationParams Optional)
     (constantly true)
     Optional)))

(s/defschema SearchParams
  "Query params for spots."
  (merge LocationParams
         PaginationParams
         {(s/optional-key :units) Units
          (s/optional-key :query) s/Str}))

(s/defschema WeatherParams
  "Query params for the weather."
  (merge PaginationParams
         {(s/optional-key :start) s/Inst
          (s/optional-key :end) s/Inst
          (s/optional-key :units) Units}))

(s/defschema WeatherForecastsParams
  "Query params for the weather forecasts."
  (merge PaginationParams
         {(s/optional-key :end) s/Inst
          (s/optional-key :latitude) Double
          (s/optional-key :longitude) Double
          (s/optional-key :start) s/Inst
          (s/optional-key :units) Units}))

(s/defschema WaveHeightParams
  "Query params for the wave height chart."
  (merge PaginationParams
         {(s/optional-key :start) s/Inst
          (s/optional-key :end) s/Inst}))

(s/defschema CommentsParams
  "Query params for comments."
  (merge PaginationParams
         {(s/optional-key :query) s/Str}))

(s/defschema EntryData
  "Schema for guestbook entry"
  {:name s/Str
   :age  Long
   :lang (s/enum :clj :cljs)})

(s/defschema Entry
  "A guestbook entry with an index"
  (assoc EntryData :index Long))

(s/defschema ClientEntry
  "Schema for a client representation of an entry"
  (-> Entry
      (dissoc :name)
      (assoc :first-name s/Str)
      (assoc :last-name s/Str)))

(s/defschema FacebookLoginParams
  "Schema for the Facebook login query params."
  {(s/optional-key :code) s/Str
   (s/optional-key :error) s/Str
   (s/optional-key :error_code) s/Str
   (s/optional-key :error_description) s/Str
   (s/optional-key :error_reason) s/Str})

(s/defschema NotFound
  {:message s/Str})

(s/defschema Forbidden
  s/Any)

(s/defn no-content
  []
  {:status 204
   :body nil})

(s/defn not-found
  [message :- s/Str]
  {:status 404
   :body {:message message}})
