(ns burningswell.api.spec.request
  (:require [clojure.spec :as s]
            [clojure.spec.gen :as gen]))

(defn number-str-gen [number-gen]
  (gen/one-of [number-gen (gen/fmap str number-gen)]))

(defn parse-integer [x]
  (cond
    (integer? x)
    x
    (string? x)
    (try
      (Integer/parseInt x)
      (catch Exception e
        (try (bigint x)
             (catch Exception e
               :clojure.spec/invalid))))
    :else :clojure.spec/invalid))

(defn parse-bool [x]
  (cond
    (boolean? x)
    x
    (string? x)
    (.booleanValue (Boolean. x))
    :else :clojure.spec/invalid))

(def not-neg?
  (complement pos-int?))

(s/def ::bool-str
  (s/with-gen (s/and (s/conformer parse-bool) boolean?)
    #(number-str-gen (s/gen boolean?))))

(s/def ::pos-int-str
  (s/with-gen (s/and (s/conformer parse-integer) pos-int?)
    #(number-str-gen (s/gen pos-int?))))

(s/def ::not-neg-int-str
  (s/with-gen (s/and (s/conformer parse-integer) pos-int?)
    #(number-str-gen (s/gen pos-int?))))

;; Query Parameters

(s/def ::page ::pos-int-str)

(s/def ::per-page ::not-neg-int-str)

(s/def ::photos ::bool-str)

(s/def ::query-params
  (s/keys :opt-un [::page ::per-page ::photos]))

(s/def ::spots
  (s/keys :req-un [::query-params]))

(defn conform!
  [spec data]
  (let [result (s/conform spec data)]
    (if (= result ::s/invalid)
      (throw (ex-info "Invalid query parameters."
                      (merge {:type :invalid-query-parameters
                              :data data
                              :errors (s/explain-data spec data)})))
      result)))
