(ns burningswell.api.weather.models
  "The weather models."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.weather :as weather]
            [burningswell.http.response :refer [created ok]]
            [datumbazo.core :refer [with-connection]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn- model-not-found
  "Return a 404 response for a region that could not be found by `id`."
  [id]
  (not-found (format "Weather model %s not found" id)))

(defapi models
  "List all weather models."
  [{:keys [api-client db query-params]}]
  (with-connection [db db]
    (let [models (weather/models db query-params)]
      (ok models))))

(defapi create-model
  "Create a new weather model."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateWeatherModel data)
  (with-connection [db db]
    (let [model (weather/create-model db data)]
      (publish broker "weather.models.created" model)
      (created model))))

(defapi model
  "Show a weather model."
  [{:keys [api-client db path-params]}]
  (with-connection [db db]
    (if-let [model (weather/model-by-id db (:id path-params))]
      (ok model)
      (model-not-found (:id path-params)))))

(defapi delete-model
  "Delete a weather model."
  [{:keys [broker db path-params]}]
  (with-connection [db db]
    (if-let [model (weather/model-by-id db (:id path-params))]
      (do (weather/delete-model db model)
          (publish broker "weather.models.deleted" model)
          (no-content))
      (model-not-found (:id path-params)))))

(defapi update-model
  "Update a weather model."
  [{:keys [api-client data broker db path-params]}]
  (with-connection [db db]
    (if-let [model (weather/model-by-id db (:id path-params))]
      (do (s/validate CreateWeatherModel data)
          (let [model (weather/update-model db (merge model data))]
            (publish broker "weather.models.updated" model)
            (ok model)))
      (model-not-found (:id path-params)))))

(set! *warn-on-reflection* false)
