(ns burningswell.api.weather.variables
  "The weather variables."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.weather :as weather]
            [burningswell.http.response :refer [created ok]]
            [datumbazo.core :refer [with-connection]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn- variable-not-found
  "Return a 404 response for a region that could not be found by `id`."
  [id]
  (not-found (format "Weather variable %s not found" id)))

(defapi variables
  "List all weather variables."
  [{:keys [api-client db query-params]}]
  (with-connection [db db]
    (let [variables (weather/variables db query-params)]
      (ok variables))))

(defapi create-variable
  "Create a new weather variable."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateWeatherVariable data)
  (with-connection [db db]
    (let [variable (weather/create-variable db data)]
      (publish broker "weather.variables.created" variable)
      (created variable))))

(defapi variable
  "Show a weather variable."
  [{:keys [api-client db path-params]}]
  (with-connection [db db]
    (if-let [variable (weather/variable-by-id db (:id path-params))]
      (ok variable)
      (variable-not-found (:id path-params)))))

(defapi delete-variable
  "Delete a weather variable."
  [{:keys [broker db path-params]}]
  (with-connection [db db]
    (if-let [variable (weather/variable-by-id db (:id path-params))]
      (do (weather/delete-variable db variable)
          (publish broker "weather.variables.deleted" variable)
          (no-content))
      (variable-not-found (:id path-params)))))

(defapi update-variable
  "Update a weather variable."
  [{:keys [api-client data broker db path-params]}]
  (with-connection [db db]
    (if-let [variable (weather/variable-by-id db (:id path-params))]
      (do (s/validate CreateWeatherVariable data)
          (let [variable (weather/update-variable db (merge variable data))]
            (publish broker "weather.variables.updated" variable)
            (ok variable)))
      (variable-not-found (:id path-params)))))

(set! *warn-on-reflection* false)
