(ns burningswell.api.sessions
  "The surf session at surf spots."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.sessions :as sessions]
            [burningswell.db.spots :as spots]
            [burningswell.http.response :refer [created ok]]
            [clj-time.core :refer [hours minus now]]
            [datumbazo.core :refer [with-connection]]
            [schema.core :as s]))

(defn- session-not-found
  "Return a 404 response for a session that could not be found by `id`."
  [id]
  (not-found (format "Session %s not found" id)))

(defapi sessions
  "List all sessions."
  [{:keys [api-client db query-params]}]
  (with-connection [db db]
    (let [sessions (sessions/sessions db query-params)]
      (ok sessions))))

(defapi create-session
  "Create a surf session for a spot."
  [{:keys [api-client data broker db identity path-params]}]
  (s/validate CreateSession data)
  (with-connection [db db]
    (when-let [spot (spots/by-id db (:spot-id data))]
      (let [started-at (or (:started-at data) (now))
            stopped-at (or (:stopped-at data) (minus started-at (hours 1)))
            session (sessions/insert-session
                     db {:user-id (:id identity)
                         :spot-id (:id spot)
                         :started-at started-at
                         :stopped-at stopped-at
                         :rating (:rating data)})]
        (publish broker "sessions.created" session)
        (created session)))))

(defapi session
  "Show a session."
  [{:keys [api-client db path-params]}]
  (with-connection [db db]
    (if-let [session (sessions/session-by-id db (:id path-params))]
      (ok session)
      (session-not-found (:id path-params)))))

(defapi delete-session
  "Delete a session."
  [{:keys [broker db path-params]}]
  (with-connection [db db]
    (if-let [session (sessions/session-by-id db (:id path-params))]
      (do (sessions/delete-session db session)
          (publish broker "sessions.deleted" session)
          (no-content))
      (session-not-found (:id path-params)))))

(defapi update-session
  "Update a session."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateSession data)
  (with-connection [db db]
    (if-let [session (sessions/session-by-id db (:id path-params))]
      (let [session (sessions/update-session db (merge session data))]
        (publish broker "sessions.updated" session)
        (ok session))
      (session-not-found (:id path-params)))))
