(ns burningswell.api.addresses
  "The addresses of users."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.addresses :as addresses]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn address-not-found
  "Return a 404 response for a address that could not be found by `id`."
  [id]
  (not-found (format "Address %s not found" id)))

(defn addresses
  "List all addresses."
  [{:keys [api-client db query-params]}]
  (let [addresses (addresses/all db query-params)
        addresses (hal/links api-client :address addresses)]
    (ok addresses)))

(defn create-address
  "Create a new address."
  [{:keys [api-client broker data db query-params]}]
  (s/validate CreateAddress data)
  (let [body (assoc-in data [:_embedded :user] identity)
        address (addresses/insert db body)
        address (hal/link api-client :address address)]
    (publish broker "addresses.created" address)
    (created address)))

(defn address
  "Show a address."
  [{:keys [api-client db path-params]}]
  (if-let [address (addresses/by-id db (:id path-params))]
    (ok (hal/link api-client :address address))
    (address-not-found (:id path-params))))

(defn delete-address
  "Delete a address."
  [{:keys [api-client broker db path-params]}]
  (if-let [address (addresses/by-id db (:id path-params))]
    (do (addresses/delete db address)
        (publish broker "addresses.deleted" address)
        (no-content))
    (address-not-found (:id path-params))))

(defn update-address
  "Update a address."
  [{:keys [api-client broker data db identity path-params]}]
  (s/validate CreateAddress data)
  (if-let [address (addresses/by-id db (:id path-params))]
    (let [data (-> (assoc data :id (:id path-params))
                   (assoc-in [:_embedded :user] identity))
          address (addresses/update db (merge address data))
          address (hal/link api-client :address address)]
      (publish broker "addresses.updated" address)
      (ok address))
    (address-not-found (:id path-params))))

(set! *warn-on-reflection* false)
