(ns burningswell.api.hal
  (:require [burningswell.api.client :refer [url-for]]
            [hal.core :refer [embedded with-embedded with-hrefs]]))

(defn- ref-country [router resource]
  (if-let [country (embedded resource :country)]
    (url-for router :country country)))

(defn- ref-photo [router resource]
  (if-let [photo (embedded resource :photo)]
    (url-for router :photo photo)))

(defn- ref-region [router resource]
  (if-let [region (embedded resource :region)]
    (url-for router :region region)))

(defn- ref-user [router resource]
  (if-let [user (embedded resource :user)]
    (url-for router :user user)))

(defn- ref-spot [router resource]
  (if-let [spot (embedded resource :spot)]
    (url-for router :spot spot)))

(defn- ref-time-zone [router resource]
  (if-let [time-zone (embedded resource :time-zone)]
    (url-for router :time-zone time-zone)))

(defmulti link (fn [_ type _] type))

(defn links [router type resources]
  (map #(link router type %) resources))

(defmethod link :address [router type address]
  (with-hrefs address
    :country (ref-country router address)
    :region (ref-region router address)
    :self (url-for router :address address)
    :user (ref-user router address)))

(defmethod link :airport [router type airport]
  (with-hrefs airport
    :self (url-for router :airport airport)
    :country (ref-country router airport)
    :region (ref-region router airport)))

(defmethod link :comment [router type comment]
  (with-hrefs comment
    :self (url-for router :comment comment)
    :spot (ref-spot router comment)
    :user (ref-user router comment)))

(defmethod link :continent [router type continent]
  (with-hrefs continent
    :airports (url-for router :airports-in-continent continent)
    :countries (url-for router :countries-in-continent continent)
    :ports (url-for router :ports-in-continent continent)
    :regions (url-for router :regions-in-continent continent)
    :self (url-for router :continent continent)
    :spots (url-for router :spots-in-continent continent)
    :users (url-for router :users-in-continent continent)))

(defmethod link :country [router type country]
  (let [continent (embedded country :continent)]
    (with-hrefs country
      :airports (url-for router :airports-in-country country)
      :continent (url-for router :continent continent)
      :photo (ref-photo router country)
      :ports (url-for router :ports-in-country country)
      :regions (url-for router :regions-in-country country)
      :self (url-for router :country country)
      :spots (url-for router :spots-in-country country)
      :users (url-for router :users-in-country country))))

(defmethod link :photo [router type photo]
  (with-hrefs photo
    :likes (url-for router :photo-likes photo)
    :self (url-for router :photo photo)
    :spot (ref-spot router photo)
    :user (ref-user router photo)))

(defmethod link :port [router type port]
  (with-hrefs port
    :self (url-for router :port port)
    :country (ref-country router port)
    :region (ref-region router port)))

(defmethod link :region [router type region]
  (with-hrefs region
    :airports (url-for router :airports-in-region region)
    :self (url-for router :region region)
    :country (ref-country router region)
    :photo (ref-photo router region)
    :ports (url-for router :ports-in-region region)
    :spots (url-for router :spots-in-region region)
    :users (url-for router :users-in-region region)))

(defmethod link :spot [router type spot]
  (-> (with-hrefs spot
        :country (ref-country router spot)
        :photo (ref-photo router spot)
        :photos (url-for router :spot-photos spot)
        :region (ref-region router spot)
        :self (url-for router :spot spot)
        :spots-around (url-for router :spots-around-spot spot)
        :user (ref-user router spot)
        :time-zone (ref-time-zone router spot)
        :weather (url-for router :spot-weather spot))
      (update-in [:_embedded :weather] #(dissoc % :_links))))

(defmethod link :time-zone [router type time-zone]
  (with-hrefs time-zone
    :self (url-for router :time-zone time-zone)))

(defmethod link :user [router type user]
  (with-hrefs user
    :self (url-for router :user user)
    :country (ref-country router user)
    :region (ref-region router user)))

(defn root
  "Return the links for the root."
  [router]
  (with-hrefs {}
    :addresses (url-for router :addresses)
    :airports (url-for router :airports)
    :comments (url-for router :comments)
    :config (url-for router :config)
    :continents (url-for router :continents)
    :countries (url-for router :countries)
    :photos (url-for router :photos)
    :ports (url-for router :ports)
    :regions (url-for router :regions)
    :spots (url-for router :spots)
    :users (url-for router :users)
    :weather-models (url-for router :weather-models)
    :weather-variables (url-for router :weather-variables)))
