(ns burningswell.api.interceptors
  (:require [burningswell.api.authentication :as authentication]
            [burningswell.api.authorization :as authorization]
            [burningswell.api.coerce :as coerce]
            [burningswell.pedestal.interceptors :as interceptor]
            [burningswell.transit :as bs-transit]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str]
            [hiccup.element :refer [javascript-tag]]
            [hiccup.page :refer [html5 include-css include-js]]
            [io.pedestal.http :as http]
            [io.pedestal.http.body-params :as body-params]
            [io.pedestal.http.content-negotiation :refer [negotiate-content]]
            [io.pedestal.interceptor :refer [interceptor]]
            [io.pedestal.interceptor.chain :refer [enqueue*]]
            [io.pedestal.interceptor.error :refer [error-dispatch]]
            [io.pedestal.interceptor.helpers :refer [on-response]]
            [ring.util.response :as ring-response]))

(def highlight-css
  "The Highlight.js stylesheet."
  (str "//cdnjs.cloudflare.com/ajax/libs/highlight.js/9.4.0"
       "/styles/solarized-light.min.css"))

(def highlight-js
  "The Highlight.js JavaScript code."
  "//cdnjs.cloudflare.com/ajax/libs/highlight.js/9.4.0/highlight.min.js")

(defn replace-links
  "Replace all URLs in `s` with an HTML anchor."
  [s]
  (str/replace s #"\"(https?://[^\"]+)\"" "\"<a href=\"$1\">$1</a>\""))

(defn pprint-body [body]
  (replace-links
   (html5
    [:head
     (include-css "/stylesheets/burningswell-api.css")
     (include-css highlight-css)
     (include-js highlight-js)
     (javascript-tag "hljs.initHighlightingOnLoad();")]
    [:body [:pre [:code (with-out-str (pprint body))]]])))

(def html-response
  (on-response
   ::html-response
   (fn [response]
     (let [body (:body response)
           content-type (get-in response [:headers "Content-Type"])]
       (if (and (coll? body) (not content-type))
         (-> response
             (ring-response/content-type "text/html;charset=UTF-8")
             (assoc :body (pprint-body body)))
         response)))))

(def accepted-content-types
  "The accepted content types."
  {"application/json"
   http/json-body
   "application/edn"
   http/edn-response
   "application/transit+json"
   (http/transit-body-interceptor
    ::transit-json-body
    "application/transit+json;charset=UTF-8"
    :json
    {:handlers bs-transit/write-handlers})
   "application/transit+msgpack"
   (http/transit-body-interceptor
    ::transit-msgpack-body
    "application/transit+msgpack;charset=UTF-8"
    :msgpack
    {:handlers bs-transit/write-handlers})
   "text/html"
   html-response})

(defn render-response
  "Return an interceptor that renders the response using the
  negotiated content type."
  [content-types]
  (interceptor
   {:name ::render-response
    :enter
    (fn [context]
      (let [content-type (-> context :request :accept :field)]
        (if-let [interceptor (get content-types content-type)]
          (enqueue* context interceptor)
          context)))}))

(def bounding-box
  "An interceptor that coerces bounding box query params."
  (interceptor
   {:name ::bounding-box
    :enter #(update-in % [:request :query-params]
                       coerce/update-bounding-box)}))

(def location
  "An interceptor that coerces bounding box query params."
  (interceptor
   {:name ::location
    :enter #(update-in % [:request :query-params] coerce/update-location)}))

(defn pagination
  "Return a pagination interceptor."
  [& [per-page]]
  (interceptor
   {:name ::pagination
    :enter
    (fn [context]
      (-> (update-in context [:request :query-params :page] #(or % 1))
          (update-in [:request :query-params :per-page]
                     #(or % per-page 10))))}))

(def zoom
  "An interceptor that coerces the zoom query params."
  (interceptor
   {:name ::zoom
    :enter #(update-in % [:request :query-params] coerce/update-zoom)}))

(def exception-interceptor
  "The exception interceptor."
  (error-dispatch
   [context exception]
   [{:exception-type :clojure.lang.ExceptionInfo}]
   (cond
     (= (:type (ex-data exception)) :schema.core/error)
     (-> (assoc-in context [:response :status] 422)
         (assoc-in [:response :body] (:error (ex-data exception))))
     :else
     (assoc context :io.pedestal.interceptor.chain/error exception))
   :else
   (assoc context :io.pedestal.interceptor.chain/error exception)))

(defn add-interceptors
  "Add the default interceptors to `server`."
  [server]
  (->> (fn [interceptors]
         (let [interceptors (interceptor/remove-interceptors-by-names
                             interceptors [:io.pedestal.http/log-request])]
           (vec (concat
                 [interceptor/log-request
                  interceptor/proxy-headers]
                 ;; Default Pedestal interceptors, except routing.
                 (butlast interceptors)
                 [(negotiate-content (keys accepted-content-types))
                  (render-response accepted-content-types)
                  exception-interceptor
                  (interceptor/dependencies server)
                  authentication/authentication-interceptor
                  (authorization/authorization-interceptor
                   {:rules authorization/access-rules
                    :on-error authorization/on-error})
                  (body-params/body-params
                   (body-params/default-parser-map
                    :transit-options [{:handlers bs-transit/read-handlers}]))
                  interceptor/merge-params-into-data]
                 ;; Pedestal routing interceptor.
                 [(last interceptors)]))))
       (update server ::http/interceptors)))
