(ns burningswell.api.parser
  (:refer-clojure :exclude [read])
  (:require [burningswell.db.countries :as countries]
            [burningswell.db.search :as search]
            [burningswell.db.spots :as spots]
            [burningswell.db.regions :as regions]
            [burningswell.db.roles :as roles]
            [burningswell.db.photos :as photos]
            [burningswell.db.users :as users]
            [burningswell.db.weather :as weather]
            [burningswell.api.schemas :refer :all]
            [clojure.string :as str]
            [clojure.pprint :refer [pprint]]
            [om.next.server :as om]
            [schema.core :as s]
            [burningswell.transit :as transit]))

(defn- add-pagination
  "Add default :page and :per-page entries to `params`. "
  [{:keys [page per-page] :as params}]
  (cond-> params
    (nil? page)
    (assoc :page 1)
    (nil? per-page)
    (assoc :per-page 10)))

(defmulti read (fn [env key params] key))

(defn- error-no-read-handler [key params]
  (format "No remote handler for read key `%s`. Params: %s"
          key params))

(defmethod read :default
  [env key params]
  (throw (ex-info (format "No read handler for key %s." key)
                  {:key key :params params} )))

(defmethod read :spots/bounding-box
  [{:keys [db]} _ params]
  (let [{:keys [bounding-box page per-page zoom] :as params}
        (add-pagination params)]
    (when bounding-box
      {:value
       (cond
         (or (nil? zoom) (< zoom 5))
         (->> (assoc params :min-spots 1)
              (countries/all db)
              (countries/assoc-photo db)
              (mapv #(assoc % :type :country)))
         (< zoom 9)
         (->> (assoc params :min-spots 1)
              (regions/all db)
              (regions/assoc-photo db)
              (mapv #(assoc % :type :region)))
         :else
         (->> (spots/all db params)
              (spots/assoc-photo db)
              (mapv #(assoc % :type :spot))))})))

(defmethod read :spots/list
  [{:keys [db]} _ params]
  (let [params (add-pagination params)]
    {:keys [:pages/spots]
     :value (vec (map #(dissoc % :location)
                      (->> (spots/all db params)
                           (spots/assoc-photo db))))}))

;; Country page

(defmethod read :country/details
  [{:keys [db]} _ {:keys [id] :as params}]
  (when-let [country (countries/by-id db id)]
    {:value country}))

(defmethod read :country/regions
  [{:keys [db]} _ {:keys [id] :as params}]
  (when-let [country (countries/by-id db id)]
    (let [params (add-pagination params)
          params (assoc params :min-spots 1)
          regions (regions/in-country db country params)
          regions (regions/assoc-photo db regions)]
      {:value regions})))

;; Region page

(defmethod read :region/details
  [{:keys [db]} _ {:keys [id] :as params}]
  (when-let [region (regions/by-id db id)]
    {:value region}))

(defmethod read :region/spots
  [{:keys [db]} _ {:keys [id] :as params}]
  (when-let [region (regions/by-id db id)]
    (let [params (add-pagination params)
          spots (spots/in-region db region params)
          spots (spots/assoc-photo db spots)]
      {:value spots})))

(defmethod read :spot/details
  [{:keys [db]} _ {:keys [id]}]
  {:value (spots/by-id db id)})

(defmethod read :autocomplete/results
  [{:keys [db]} _ {:keys [page per-page query] :as params}]
  (let [params (add-pagination params)]
    (when-not (str/blank? query)
      {:value (search/autocomplete db params)})))

;; Spot page

(defmethod read :spot/spots-around
  [{:keys [db]} _ {:keys [id] :as params}]
  (let [params (add-pagination params)]
    (when-let [spot (spots/by-id db id)]
      {:value (->> (spots/around-spot db spot params)
                   (spots/assoc-photo db))})))

(defmethod read :spot/weather
  [{:keys [db]} _ {:keys [id] :as params}]
  ;; (let [params (add-pagination params)
  ;;       spot (spots/by-id db id)]
  ;;   {:value (weather/weather-by-spot db spot)})
  {:value nil})

(defmethod read :spot/photos
  [{:keys [db]} _ {:keys [id] :as params}]
  (let [params (add-pagination params)
        spot (spots/by-id db id)]
    {:value (photos/by-spot db spot {:images true})}))

(defmethod read :header/search
  [{:keys [parser query] :as env} _ _]
  {:value (parser env query)})

(defmethod read :page/index
  [{:keys [db env parser]} _ {:keys [id] :as params}]
  {:value nil})

(defmethod read :page/country
  [{:keys [parser query] :as env} _ _]
  {:value (parser env query)})

(defmethod read :page/map
  [{:keys [parser query] :as env} _ _]
  {:value (parser env query)})

(defmethod read :page/region
  [{:keys [parser query] :as env} _ _]
  {:value (parser env query)})

(defmethod read :page/layout
  [{:keys [parser query] :as env} _ _]
  {:value (parser env query)})

(defmethod read :page/signin
  [{:keys [parser query] :as env} _ _]
  {:value (parser env query)})

(defmethod read :page/signup
  [{:keys [parser query] :as env} _ _]
  {:value (parser env query)})

(defmethod read :page/spot
  [{:keys [db parser query target] :as env} _ _]
  {:value (parser env query target)})

(defmethod read :page/spots
  [{:keys [db parser query target] :as env} _ _]
  {:value (parser env query target)})

(defmulti mutate (fn [env key params] key))

(defmethod mutate 'signin/submit
  [{:keys [db]} _ {:keys [username password] :as params}]
  {:action #(users/authenticate-password db params)
   :value {:keys [:signin]}})

(defmethod mutate 'signup/submit
  [{:keys [db]} _ {:keys [email username password] :as params}]
  (if-let [errors (s/check (CreateUser db) params)]
    {:value {:errors errors}}
    (let [user (users/insert db params)]
      (users/add-to-role db user (roles/surfer db))
      {:value user})))

(defmethod mutate :default
  [env key params]
  (throw  (ex-info (format "No mutation handler for key %s." key)
                   {:key key :params params})))

(def parser
  "The Om.next parser."
  (om/parser {:read read :mutate mutate}))
