(ns burningswell.api.weather.datasets
  "The weather datasets."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.weather :as weather]
            [burningswell.http.response :refer [created ok]]
            [datumbazo.core :refer [with-connection]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn- dataset-not-found
  "Return a 404 response for a region that could not be found by `id`."
  [id]
  (not-found (format "Weather dataset %s not found" id)))

(defn datasets
  "List all weather datasets."
  [{:keys [api-client db query-params]}]
  (with-connection [db db]
    (let [datasets (weather/datasets db query-params)]
      (ok datasets))))

(defn create-dataset
  "Create a new weather dataset."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateWeatherDataset data)
  (with-connection [db db]
    (let [dataset (weather/create-dataset db data)]
      (publish broker "weather.datasets.created" dataset)
      (created dataset))))

(defn dataset
  "Show a weather dataset."
  [{:keys [api-client db path-params]}]
  (with-connection [db db]
    (if-let [dataset (weather/dataset-by-id db (:id path-params))]
      (ok dataset)
      (dataset-not-found (:id path-params)))))

(defn delete-dataset
  "Delete a weather dataset."
  [{:keys [broker db path-params]}]
  (with-connection [db db]
    (if-let [dataset (weather/dataset-by-id db (:id path-params))]
      (do (weather/delete-dataset db dataset)
          (publish broker "weather.datasets.deleted" dataset)
          (no-content))
      (dataset-not-found (:id path-params)))))

(defn update-dataset
  "Update a weather dataset."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateWeatherDataset data)
  (with-connection [db db]
    (if-let [dataset (weather/dataset-by-id db (:id path-params))]
      (let [dataset (weather/update-dataset db (merge dataset data))]
        (publish broker "weather.datasets.updated" dataset)
        (ok dataset))
      (dataset-not-found (:id path-params)))))

(set! *warn-on-reflection* false)
