(ns burningswell.cli.spots.import
  (:require [burningswell.api.client :as api]
            [burningswell.file :refer [edn-line-seq]]
            [clojure.java.io :as io]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [commandline.core :as cli]
            [environ.core :refer [env]]
            [request.core :as http]))

(defn api-client
  "Return a new API client."
  [env opts]
  (api/new-client
   {:basic-auth [(:username opts) (:password opts)]
    :scheme (or (:scheme opts) (:bs-api-scheme env) "http")
    :server-name (or (:server opts) (:bs-api-server-name env) "localhost")
    :server-port (or (:port opts) (:bs-api-server-port env) 8001)}))

(defn assert-status-msg
  "Assert that the :status code of `response` is `expected`."
  [{:keys [status]} expected]
  (str "Expected HTTP status "
       (if (sequential? expected)
         (str/join ", " expected) expected)
       ", but got " status "."))

(defn assert-status-error
  "Assert that the :status code of `response` is `expected`."
  [response expected]
  (ex-info (assert-status-msg response expected) response))

(defn assert-status
  "Assert that the :status code of `response` is `expected`."
  [{:keys [status] :as response} expected]
  (cond
    (and (number? expected)
         (not= expected status))
    (throw (assert-status-error response expected))
    (and (set? expected)
         (not (contains? expected status)))
    (throw (assert-status-error response expected)))
  response)

(defn create-spot
  "Create a new spot."
  [api-client spot & [opts]]
  (clojure.pprint/pprint (http/request-for api-client :spots (assoc opts :form-params spot)))
  (-> (http/post api-client :spots (assoc opts :form-params spot))
      (assert-status 201)
      :body))

(defn import-spots
  "Import all spots in EDN `files`."
  [files opts]
  (let [api-client (api-client env opts)]
    (doseq [file files]
      (with-open [reader (io/reader file)]
        (doseq [spot (edn-line-seq reader)]
          (try (let [created (create-spot api-client spot)
                     region (-> created :_embedded :region :name)
                     country (-> created :_embedded :country :name)]
                 (log/info
                  (str "Created spot #" (:id created) " "
                       (:name created)
                       (when (or region country) " in ")
                       (when region region)
                       (when (and region country) ", ")
                       (when country country) ".")))
               (catch Exception e
                 (if-let [status (-> e ex-data :status)]
                   (log/errorf "Can't create spot %s. HTTP status %s.\n%s"
                               (:name spot) status
                               (with-out-str (pprint (-> e ex-data :body))))
                   (throw e)))))))))

(defn -main [& args]
  (cli/with-commandline [[opts args] args]
    [[h help "Print this help."]
     [s server "Use SERVER as API host name." :string "SERVER"]
     [S scheme "Use SCHEME as the API HTTP scheme." :string "SCHEME"]
     [P port "Use PORT as API port number." :integer "PORT"]
     [u username "Use USERNAME with HTTP basic authentication." :string "USERNAME"]
     [p password "Use PASSWORD with HTTP basic authentication." :string "PASSWORD"]]
    (if (or (:help opts) (empty? args))
      (cli/print-help "bs-cli spots import [OPTIONS...] FILE")
      (import-spots args opts))))
