(ns burningswell.cli.weather.import
  (:gen-class)
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :as db]
            [burningswell.db.weather :as weather]
            [burningswell.db.weather.models :as models]
            [burningswell.db.weather.variables :as variables]
            [burningswell.time :refer [current-duration-str]]
            [clj-time.coerce :refer [to-date-time]]
            [clj-time.core :refer [day hour month now year]]
            [clojure.contrib.humanize :as humanize]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [commandline.core :as cli]
            [datumbazo.core :as sql]
            [datumbazo.shell :as shell :refer [raster2pgsql]]
            [datumbazo.util :refer [exec-sql-file]]
            [environ.core :refer [env]]
            [netcdf.dataset :as dataset]
            [netcdf.dods :as dods]
            [no.en.core :refer [split-by-comma]]))

(defn datasets-by-config
  "Return the datasets to be imported from `config`."
  [db {:keys [start end reference-time models variables] :as config}]
  (let [models (set (models/by-names-or-all db models))
        variables (variables/by-names-or-all db variables)]
    (for [variable (sort-by :name variables)
          model (filter models (models/by-variable db variable))]
      {:end end
       :model model
       :reference-time reference-time
       :start start
       :variable variable})))

(defn resolve-dataset
  "Lookup the dataset in the DODS registry."
  [{:keys [variable model reference-time] :as dataset}]
  (let [reference-time (or reference-time (now))]
    (if-let [datasource (dods/datasource model reference-time)]
      (dataset/with-grid-dataset [grid (:dods datasource)]
        (let [valid-times (dataset/valid-times grid)]
          (log/infof "Importing dataset:")
          (log/infof "  Variable .......... %s (%s)"
                     (:description variable) (:name variable))
          (log/infof "  Model ............. %s (%s)"
                     (:description model) (:name model))
          (log/infof "  Reference Time .... %s"
                     (:reference-time datasource))
          (log/infof "  Time Range  ....... %s - %s"
                     (first valid-times) (last valid-times))
          (assoc  dataset
                  :das (:das datasource)
                  :dds (:dds datasource)
                  :dods (:dods datasource)
                  :reference-time (:reference-time datasource)
                  :valid-times valid-times)))
      (throw (ex-info "Can't open dataset."
                      {:model model
                       :variable variable
                       :reference-time reference-time})))))

(defn netcdf-file
  "Return the NetCDF file for `dataset`."
  [{:keys [directory variable model reference-time] :as dataset}]
  (io/file
   (or directory "netcdf")
   (:name model)
   (:name variable)
   (format "%04d" (year reference-time))
   (format "%02d" (month reference-time))
   (format "%02d" (day reference-time))
   (format "%02d.nc" (hour reference-time))))

(defn netcdf-checksum-file
  "Return the NetCDF checksum file for `dataset`."
  [dataset]
  (-> (netcdf-file dataset) (str ".md5") io/file))

(defn sql-file
  "Return the SQL file for `dataset`."
  [dataset]
  (-> (netcdf-file dataset) (str/replace #".nc" ".sql") (io/file)))

(defn raster-table
  "Returns the temporary raster table for `dataset`."
  [{:keys [model variable reference-time] :as dataset}]
  (keyword (format "%s-%s-%04d-%02d-%02d-%02d"
                   (:name model) (:name variable)
                   (year reference-time)
                   (month reference-time)
                   (day reference-time)
                   (hour reference-time))))

(defn download-dataset
  "Download the dataset."
  [{:keys [dods model variable reference-time valid-times] :as dataset}]
  (let [netcdf (netcdf-file dataset)
        checksum (netcdf-checksum-file dataset)]
    (io/make-parents netcdf)
    (when-not (.exists checksum)
      (dataset/copy-dataset dods netcdf [(:name variable)]))
    (log/infof "  NetCDF File Name .. %s" (str netcdf))
    (log/infof "  NetCDF File Size .. %s" (humanize/filesize (.length netcdf)))
    (assoc dataset
           :netcdf-checksum-file (str checksum)
           :netcdf-file (str netcdf))))

(defn create-raster-sql
  "Create the raster SQL file."
  [db {:keys [netcdf-file] :as dataset}]
  (let [sql-file (sql-file dataset)
        tmp-table (raster-table dataset)]
    (raster2pgsql db tmp-table netcdf-file (str sql-file)
                  {:constraints true
                   :height 100
                   :mode :create
                   :padding true
                   :regular-blocking true
                   :srid 4326
                   :width 100})
    (log/infof "  SQL File Name ..... %s" (str sql-file))
    (log/infof "  SQL File Size ..... %s" (humanize/filesize (.length sql-file)))
    (assoc dataset
           :sql-file (str sql-file)
           :tmp-table tmp-table)))

(defn save-dataset
  "Save the `dataset` to `db`."
  [db dataset valid-time]
  (first @(sql/insert db :weather.datasets []
            (sql/values [{:das (:das dataset)
                          :dds (:dds dataset)
                          :dods (:dods dataset)
                          :model-id (-> dataset :model :id)
                          :reference-time (:reference-time dataset)
                          :valid-time valid-time
                          :variable-id (-> dataset :variable :id)}])
            (sql/on-conflict [:model-id :variable-id :valid-time]
              (sql/do-update {:reference-time :EXCLUDED.reference-time}))
            (sql/returning :*))))

(defn delete-raster
  "Delete the existing raster."
  [db dataset]
  @(sql/delete db :weather.rasters
     (sql/where `(= :dataset-id ~(:id dataset)))))

(defn insert-raster
  "Insert the new raster."
  [db dataset table band]
  (let [column (keyword (str (name table) ".rast"))]
    @(sql/insert db :weather.rasters [:dataset-id :rast]
       (sql/select db [(sql/as (:id dataset) :dataset-id)
                       `(st_band ~column ~band)]
         (sql/from table)
         (sql/join :weather.rasters
                   `(on (= :weather.rasters.dataset-id
                           ~(:id dataset)))
                   :type :left)
         (sql/where '(is-null :weather.rasters.id))))))

(defn load-raster-sql
  "Load the raster SQL file."
  [db {:keys [sql-file tmp-table valid-times] :as dataset}]
  @(sql/drop-table db [tmp-table] (sql/if-exists true))
  (exec-sql-file db sql-file)
  (doseq [[index valid-time] (map-indexed vector valid-times)
          :let [dataset (save-dataset db dataset valid-time)]]
    (delete-raster db dataset)
    (insert-raster db dataset tmp-table (inc index)))
  @(sql/drop-table db [tmp-table])
  dataset)

(defn create-forecast
  "Create the spot forecasts."
  [db {:keys [model variable start end valid-times] :as dataset}]
  (let [started-at (now)
        results @(weather/import-spot-forecasts
                  db {:models [model]
                      :variables [variable]
                      :start (or start (first valid-times))
                      :end (or end (last valid-times))})]
    (log/infof "  Forecast Rows ..... %s (%s)"
               (-> results first :count)
               (current-duration-str started-at))
    dataset))

(defn existing-dataset?
  "Returns true if `dataset` exists in `db`, otherwise false."
  [db {:keys [model variable reference-time] :as dataset}]
  (-> @(sql/select db ['(count :*)]
         (sql/from :weather.datasets)
         (sql/where `(and (= :datasets.model-id ~(:id model))
                          (= :datasets.variable-id ~(:id variable))
                          (= :datasets.reference-time ~reference-time))))
      first :count pos?))

(defn make-pipeline
  "Return the import pipeline."
  [db]
  (comp
   (map resolve-dataset)
   (remove (partial existing-dataset? db))
   (map download-dataset)
   (map (partial create-raster-sql db))
   (map (partial load-raster-sql db))
   (map (partial create-forecast db))))

(defn run
  "Import the weather datasets and update the forecasts."
  [db config]
  (log/infof "Importing weather data ...")
  (let [started-at (now)
        pipeline (make-pipeline db)
        datasets (into [] pipeline (datasets-by-config db config))]
    (log/infof "Weather data successfully imported in %s."
               (current-duration-str started-at))
    datasets))

(defn -main [& args]
  (cli/with-commandline [[opts variables] args]
    [[e end "The weather forecast end time." :string "END"]
     [h help "Print this help."]
     [d directory "The directory where NetCDF files are stored." :string "DIRECTORY"]
     [m models "The weather models to load." :string "MODELS"]
     [r reference-time "The reference time to load." :time "REFERENCE-TIME"]
     [s start "The weather forecast start time." :time "START"]]
    (if (:help opts)
      (cli/print-help "bs-cli weather import [OPTION...] [VARIABLE,...]")
      (db/with-db [db (config/db env)]
        (sql/with-connection [db db]
          (run db {:end (to-date-time (:end opts))
                   :models (set (split-by-comma (:models opts)))
                   :reference-time (to-date-time (:reference-time opts))
                   :start (to-date-time (:start opts))
                   :variables (set variables)}))))))

(comment
  (weather/import-spot-forecasts (db))
  (-main)
  (-main "-s" "2016-03-25" "-e" "2016-03-27" "-m" "nww3" "htsgwsfc")
  (-main "-r" "2016-12-02T06:00:00.000-00:00" "-m" "nww3" "htsgwsfc")
  (-main "-r" "2016-12-03T06:00:00.000-00:00" "-m" "nww3" "htsgwsfc")
  (-main "-m" "nww3,gfs")
  (-main "-m" "nww3" "htsgwsfc")
  (-main "-m" "nww3" "htsgwsfc")
  (-main "-m" "nww3")
  (-main "-m" "gfs")
  (-main))
