(ns burningswell.cli.weather.import
  (:gen-class)
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :as db]
            [burningswell.db.weather :as weather]
            [burningswell.db.weather.models :as models]
            [burningswell.db.weather.variables :as variables]
            [burningswell.time :refer [format-interval]]
            [clj-time.coerce :refer [to-date-time]]
            [clj-time.core :refer [day hour interval month now year]]
            [clojure.contrib.humanize :refer [filesize]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [commandline.core :as cli]
            [datumbazo.core :as sql]
            [datumbazo.shell :as shell :refer [raster2pgsql]]
            [datumbazo.util :refer [exec-sql-file]]
            [environ.core :refer [env]]
            [netcdf.dataset :as dataset]
            [netcdf.dods :as dods]
            [no.en.core :refer [split-by-comma]]))

(defn datasets-by-config
  "Return the datasets to be imported from `config`."
  [db {:keys [start end reference-time models variables] :as config}]
  (let [models (set (models/by-names-or-all db models))]
    (for [variable (sort-by :name (variables/by-names-or-all db variables))
          model (filter models (models/by-variable db variable))]
      {:end end
       :model model
       :reference-time reference-time
       :start start
       :variable variable})))

(defn resolve-dataset
  "Lookup the dataset in the DODS registry."
  [{:keys [variable model reference-time] :as dataset}]
  (let [reference-time (or reference-time (now))]
    (if-let [datasource (dods/datasource model reference-time)]
      (dataset/with-grid-dataset [grid (:dods datasource)]
        (let [valid-times (dataset/valid-times grid)]
          (assoc dataset
                 :das (:das datasource)
                 :dds (:dds datasource)
                 :dods (:dods datasource)
                 :reference-time (:reference-time datasource)
                 :valid-times valid-times)))
      (throw (ex-info "Can't open dataset."
                      {:model model
                       :variable variable
                       :reference-time reference-time})))))

(defn netcdf-file
  "Return the NetCDF file for `dataset`."
  [{:keys [directory variable model reference-time] :as dataset}]
  (io/file
   (or directory "netcdf")
   (:name model)
   (:name variable)
   (format "%04d" (year reference-time))
   (format "%02d" (month reference-time))
   (format "%02d" (day reference-time))
   (format "%02d.nc" (hour reference-time))))

(defn netcdf-checksum-file
  "Return the NetCDF checksum file for `dataset`."
  [dataset]
  (-> (netcdf-file dataset) (str ".md5") io/file))

(defn sql-file
  "Return the SQL file for `dataset`."
  [dataset]
  (-> (netcdf-file dataset) (str/replace #".nc" ".sql") (io/file)))

(defn raster-table
  "Returns the temporary raster table for `dataset`."
  [{:keys [model variable reference-time] :as dataset}]
  (keyword (format "%s-%s-%04d-%02d-%02d-%02d"
                   (:name model) (:name variable)
                   (year reference-time)
                   (month reference-time)
                   (day reference-time)
                   (hour reference-time))))

(defn download-dataset
  "Download the dataset."
  [{:keys [dods model variable reference-time valid-times] :as dataset}]
  (let [started-at (now)
        file (netcdf-file dataset)
        checksum (netcdf-checksum-file dataset)
        existing? (.exists checksum)]
    (when-not existing?
      (io/make-parents file)
      (dataset/copy-dataset dods file [(:name variable)]))
    (->> {:checksum checksum
          :duration (interval started-at (now))
          :file file
          :needed? (not existing?)}
         (assoc dataset :download))))

(defn create-raster-sql
  "Create the raster SQL file."
  [db dataset]
  (let [input (-> dataset :download :file)
        file (sql-file dataset)
        table (raster-table dataset)]
    (raster2pgsql
     db table input (str file)
     {:constraints true
      :height 100
      :mode :create
      :padding true
      :regular-blocking true
      :srid 4326
      :width 100})
    (->> {:file file
          :table table}
         (assoc dataset :load))))

(defn save-datasource
  "Save the datasource to `db`."
  [db {:keys [model variable reference-time valid-times] :as dataset}]
  (->> @(sql/insert db :weather.datasources []
          (sql/values [{:das (:das dataset)
                        :dds (:dds dataset)
                        :dods (:dods dataset)
                        :model-id (-> dataset :model :id)
                        :reference-time (:reference-time dataset)}])
          (sql/on-conflict [:model-id :reference-time]
            (sql/do-update {:das :EXCLUDED.das
                            :dds :EXCLUDED.dds
                            :dods :EXCLUDED.dods}))
          (sql/returning :*))
       (first)
       (assoc dataset :datasource)))

(defn insert-dataset
  "Insert the `dataset` for `valid-time` into `db`."
  [db dataset valid-time]
  (first @(sql/insert db :weather.datasets []
            (sql/values [{:datasource-id (-> dataset :datasource :id)
                          :valid-time valid-time
                          :variable-id (-> dataset :variable :id)}])
            (sql/on-conflict [:datasource-id :variable-id :valid-time]
              (sql/do-nothing))
            (sql/returning :*))))

(defn select-dataset
  "Find the `dataset` for `valid-time` in `db`."
  [db dataset valid-time]
  (first @(sql/select db [:*]
            (sql/from :weather.datasets)
            (sql/where `(and (= :datasource-id ~(-> dataset :datasource :id))
                             (= :valid-time ~valid-time)
                             (= :variable-id ~(-> dataset :variable :id))) ))))

(defn save-dataset
  "Save the dataset to `db`."
  [db dataset valid-time]
  (->> (or (insert-dataset db dataset valid-time)
           (select-dataset db dataset valid-time))
       (assoc dataset :dataset)))

(defn delete-raster
  "Delete the existing raster."
  [db dataset]
  @(sql/delete db :weather.rasters
     (sql/where
      `(and (= :model-id ~(-> dataset :model :id))
            (= :variable-id ~(-> dataset :variable :id))
            (= :valid-time ~(:valid-time dataset))))))

(defn insert-raster
  "Insert the new raster."
  [db dataset table band]
  (let [column (keyword (str (name table) ".rast"))]
    @(sql/insert db :weather.rasters
         [:dataset-id :model-id :variable-id
          :valid-time :reference-time :rast]
       (sql/select db [(-> dataset :dataset :id)
                       (-> dataset :model :id)
                       (-> dataset :variable :id)
                       (-> dataset :dataset :valid-time)
                       (-> dataset :datasource :reference-time)
                       `(st_band ~column ~band)]
         (sql/from table)))))

(defn load-raster-sql
  "Load the raster SQL file."
  [db {:keys [valid-times] :as dataset}]
  (let [started-at (now), {:keys [file table]} (:load dataset)]
    @(sql/drop-table db [table]
       (sql/if-exists true))
    (exec-sql-file db file)
    (doseq [[index valid-time] (map-indexed vector valid-times)
            :let [dataset (save-dataset db dataset valid-time)]]
      (delete-raster db dataset)
      (insert-raster db dataset table (inc index)))
    @(sql/drop-table db [table])
    (assoc-in dataset [:load :duration] (interval started-at (now)))))

(defn create-forecast
  "Create the spot forecasts."
  [db {:keys [model variable start end valid-times] :as dataset}]
  (let [started-at (now)
        results @(weather/import-spot-forecasts
                  db {:models [model]
                      :variables [variable]
                      :start (or start (first valid-times))
                      :end (or end (last valid-times))})]
    (->> {:duration (interval started-at (now))
          :rows (-> results first :count)}
         (assoc dataset :forecast))))

(defn existing-dataset?
  "Returns true if `dataset` exists in `db`, otherwise false."
  [db {:keys [model variable reference-time] :as dataset}]
  (-> @(sql/select db ['(count :*)]
         (sql/from :weather.datasources)
         (sql/join :weather.datasets.datasource-id :weather.datasources.id)
         (sql/where `(and (= :datasources.model-id ~(:id model))
                          (= :datasources.reference-time ~reference-time)
                          (= :datasets.variable-id ~(:id variable)))))
      first :count pos?))

(defn log-summary
  "Log the import of `dataset`."
  [{:keys [model reference-time valid-times variable] :as dataset}]
  (log/infof "Imported dataset:")
  (log/infof "  Variable .............. %s (%s)"
             (:description variable) (:name variable))
  (log/infof "  Model ................. %s (%s)"
             (:description model) (:name model))
  (log/infof "  Reference Time ........ %s" reference-time)
  (log/infof "  Time Range  ........... %s - %s"
             (first valid-times) (last valid-times))

  (let [{:keys [duration file needed?]} (:download dataset)]
    (when needed?
      (log/infof "  Download Duration ..... %s" (format-interval duration)))
    (log/infof "  NetCDF File Name ...... %s" file)
    (log/infof "  NetCDF File Size ...... %s" (filesize (.length file))))

  (let [{:keys [duration file]} (:load dataset)]
    (log/infof "  SQL File Name ......... %s" (str file))
    (log/infof "  SQL File Size ......... %s" (filesize (.length file)))
    (log/infof "  SQL Load Duration...... %s" (format-interval duration)))

  (let [{:keys [duration rows]} (:forecast dataset)]
    (log/infof "  Forecast Rows ......... %s " rows)
    (log/infof "  Forecast Duration...... %s " (format-interval duration)))
  dataset)

(defn pipeline
  "Return the import pipeline."
  [db]
  (comp
   (map resolve-dataset)
   (remove (partial existing-dataset? db))
   (map (partial save-datasource db))
   (map download-dataset)
   (map (partial create-raster-sql db))
   (map (partial load-raster-sql db))
   (map (partial create-forecast db))
   (map log-summary)))

(defn run
  "Import the weather datasets and update the forecasts."
  [db config]
  (let [started-at (now)
        datasets (into [] (pipeline db) (datasets-by-config db config))]
    (when-not (empty? datasets)
      (log/infof "Weather data successfully imported in %s."
                 (format-interval (interval started-at (now)))))
    datasets))

(defn -main [& args]
  (cli/with-commandline [[opts variables] args]
    [[e end "The weather forecast end time." :string "END"]
     [h help "Print this help."]
     [d directory "The directory where NetCDF files are stored." :string "DIRECTORY"]
     [m models "The weather models to load." :string "MODELS"]
     [r reference-time "The reference time to load." :time "REFERENCE-TIME"]
     [s start "The weather forecast start time." :time "START"]]
    (if (:help opts)
      (cli/print-help "bs-cli weather import [OPTION...] [VARIABLE,...]")
      (db/with-db [db (config/db env)]
        (sql/with-connection [db db]
          (run db {:end (to-date-time (:end opts))
                   :models (set (split-by-comma (:models opts)))
                   :reference-time (to-date-time (:reference-time opts))
                   :start (to-date-time (:start opts))
                   :variables (set variables)}))))))

(comment
  (-main "-s" "2016-03-25" "-e" "2016-03-27" "-m" "nww3" "htsgwsfc")
  (-main "-r" "2016-12-02T06:00:00.000-00:00" "-m" "nww3" "htsgwsfc")
  (-main "-r" "2016-12-03T06:00:00.000-00:00" "-m" "nww3" "htsgwsfc")
  (-main "-m" "nww3,gfs")
  (-main "-m" "nww3" "htsgwsfc")
  (-main "-m" "nww3")
  (-main "-m" "gfs")
  (-main))
