(ns burningswell.cli.countries.import
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :refer [with-db]]
            [burningswell.db.continents :as continents]
            [burningswell.db.countries :as countries]
            [clojure.tools.logging :as log]
            [commandline.core :as cli]
            [environ.core :refer [env]]
            [geonames.countries :as geonames]))

(defn save-country
  "Save a country from geonames.org."
  [db country]
  (if-let [continent (continents/by-code db (:continent-code country))]
    (if-let [found (or (countries/by-name db (:name country))
                       (countries/by-iso-3166-1-alpha-2
                        db (:iso-3166-1-alpha-2 country)))]
      (->> (assoc (merge found country)
                  :id (:id found)
                  :_embedded {:continent continent})
           (countries/update db))
      (->> (assoc-in country [:_embedded :continent] continent)
           (countries/insert db)))
    (log/warnf "Can't find continent by code: %s" (:continent-code country))))

(defn import-countries
  "Import countries from geonames.org."
  [db & [countries]]
  (doseq [country (or countries (geonames/countries))]
    (try (save-country db country)
         (log/infof "Imported country: %s" (:name country))
         (catch Exception e
           (log/warnf e "Can't import country: %s" (:name country))))))

(defn -main [& args]
  (cli/with-commandline [[opts [cmd & args]] args]
    [[h help "Print this help."]]
    (when (:help opts)
      (cli/print-help "geonames"))
    (with-db [db (config/db env)]
      (import-countries db))))
