(ns burningswell.cli.natural-earth
  (:gen-class)
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :refer [with-db]]
            [burningswell.db.spots :as spots]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [commandline.core :refer [print-help with-commandline]]
            [datumbazo.core :as sql]
            [datumbazo.shell :as shell]
            [datumbazo.util :as util]
            [environ.core :refer [env]]
            [sqlingvo.util :refer [sql-name]])
  (:import org.apache.commons.io.FileUtils))

(def default-working-dir
  "The working directory for importing natural earth data."
  "natural-earth")

(def natural-earth-10m-cultural
  "The URL of the Natural Earth 10m cultural datasets."
  (str "http://www.naturalearthdata.com/http//www.naturalearthdata.com"
       "/download/10m/cultural"))

(def ^:dynamic *airports*
  "The Natural Earth airport dataset."
  {:url (str natural-earth-10m-cultural "/ne_10m_airports.zip")
   :table :natural-earth.airports
   :encoding "LATIN1"})

(def ^:dynamic *countries*
  "The Natural Earth country dataset."
  {:url (str natural-earth-10m-cultural "/ne_10m_admin_0_countries.zip")
   :table :natural-earth.countries
   :encoding "UTF-8"})

(def ^:dynamic *ports*
  "The Natural Earth port dataset."
  {:url (str natural-earth-10m-cultural "/ne_10m_ports.zip")
   :table :natural-earth.ports
   :encoding "UTF-8"})

(def ^:dynamic *regions*
  "The Natural Earth region dataset."
  {:url (str natural-earth-10m-cultural "/ne_10m_admin_1_states_provinces.zip")
   :table :natural-earth.regions
   :encoding "UTF-8"})

(def ^:dynamic *time-zones*
  "The Natural Earth time zone dataset."
  {:url (str natural-earth-10m-cultural "/ne_10m_time_zones.zip")
   :table :natural-earth.time-zones
   :encoding "LATIN1"})

(def ^:dynamic *datasets*
  "The Natural Earth datasets."
  [*countries*
   *regions*
   *airports*
   *ports*
   *time-zones*])

(defmulti insert-dataset
  "Insert the Natural Earth `dataset` into the database."
  (fn [db dataset] (:table dataset)))

(defmulti update-dataset
  "Update the Natural Earth `dataset` into the database."
  (fn [db dataset] (:table dataset)))

(defn download-dataset
  "Download the Natural Earth `dataset`."
  [dataset & [opts]]
  (let [working-dir (or (:working-dir opts) default-working-dir)
        zipfile (io/file working-dir (shell/basename (:url dataset)))]
    (io/make-parents zipfile)
    (log/infof "Downloading Natural Earth dataset." )
    (log/infof "  Dataset Url ...... %s" (:url dataset))
    (log/infof "  File Name ........ %s" (str zipfile))
    (shell/exec-checked-script
     (format "Downloading %s to %s" (:url dataset) zipfile)
     ("wget" "-c" "-O" ~(str zipfile) ~(:url dataset)))
    (assoc dataset :zip (.getAbsolutePath zipfile))))

(defn unzip-dataset
  "Unzip the downloaded Natural Earth `dataset`."
  [dataset]
  (log/infof "Unzipping Natural Earth dataset.")
  (log/infof "  File Name ........ %s" (:zip dataset))
  (shell/exec-checked-script
   (format "Unzipping %s" (:zip dataset))
   ("pushd" ~(shell/dirname (:zip dataset)))
   ("unzip" "-o" ~(shell/basename (:zip dataset)))
   ("pushd"))
  dataset)

(defn load-dataset
  "Load the Natural Earth `dataset` into the database."
  [db dataset & {:keys [entities]}]
  (let [shape (str/replace (:zip dataset) #"\.zip" ".dbf")
        sql (str/replace (:zip dataset) #"\.zip" ".sql")
        table (sql-name db (:table dataset))]
    (log/infof "Converting Natural Earth dataset shapefile to SQL.")
    (log/infof "  Shape File Name .. %s" shape)
    (log/infof "  SQL File Name .... %s" sql)
    @(sql/drop-table db [(:table dataset)]
       (sql/if-exists true))
    (shell/shp2pgsql
     db table shape sql
     {:mode :create
      :index true
      :srid 4326
      :encoding (:encoding dataset)})
    (log/infof "Loading Natural Earth SQL file." sql table)
    (log/infof "  Database Table ... %s" (name table))
    (log/infof "  SQL File Name .... %s" sql)
    (util/exec-sql-file db sql)
    (assoc dataset :shape shape :sql sql)))

(defn import-dataset
  "Import the Natural Earth `dataset` into the database."
  [db dataset & [opts]]
  (->> (download-dataset dataset opts)
       (unzip-dataset)
       (load-dataset db)
       (update-dataset db)
       (insert-dataset db)))

(defn import-natural-earth
  "Import the Natural Earth dataset into the database."
  [db & [opts]]
  (doseq [dataset *datasets*]
    (import-dataset db dataset opts)))

;; BULK INSERT

(defmethod insert-dataset :natural-earth.airports [db dataset]
  @(sql/insert db :airports [:country-id :region-id :name :gps-code
                             :iata-code :wikipedia-url :location]
     (sql/select db (sql/distinct
                     [:r.country-id :r.id :a.name :a.gps_code
                      :a.iata_code :a.wikipedia :a.geom]
                     :on [:a.iata_code])
       (sql/from (sql/as :natural-earth.airports :a))
       (sql/join (sql/as :regions :r)
                 '(on (st_intersects
                       :r.geom
                       (cast :a.geom :geography))))
       (sql/join :airports '(on (= (lower :airports.iata-code)
                                   (lower :a.iata_code)))
                 :type :left)
       (sql/where '(and (is-not-null :a.gps_code)
                        (is-not-null :a.iata_code)
                        (is-null :airports.iata-code)))))
  dataset)

(defmethod insert-dataset :natural-earth.ports [db dataset]
  @(sql/insert db :ports [:country-id :region-id :name :type
                          :website-url :location]
     (sql/select db (sql/distinct
                     [:r.country-id :r.id :p.name :p.featurecla
                      :p.website :p.geom]
                     :on [:p.name])
       (sql/from (sql/as :natural-earth.ports :p))
       (sql/join (sql/as :regions :r)
                 '(on (st_intersects :r.geom (cast :p.geom :geography))))
       (sql/join :ports
                 '(on (= (lower :ports.name) (lower :p.name)))
                 :type :left)
       (sql/where '(is-null :ports.name))))
  dataset)

(defmethod insert-dataset :natural-earth.regions [db dataset]
  @(sql/insert db :regions [:country-id :name :geom :location]
     (sql/select db [:countries.id :n.name :n.geom '(st_centroid :n.geom)]
       (sql/from (sql/as :natural-earth.regions :n))
       (sql/join :countries
                 '(on (= (lower :countries.iso-3166-1-alpha-2)
                         (lower (substring :iso_3166_2 from 1 for 2)))))
       (sql/join :regions
                 '(on (and (= (lower :regions.name) (lower :n.name))
                           (= :regions.country-id :countries.id)))
                 :type :left)
       (sql/where '(and (is-not-null :n.name)
                        (is-null :regions.id)))))
  dataset)

(defmethod insert-dataset :natural-earth.time-zones [db dataset]
  @(sql/insert db :time-zones [:geom :natural-earth-id :offset :places]
     (sql/select db [:n.geom :n.gid :n.zone
                     '(regexp_split_to_array :n.places "\\s*,\\s*")]
       (sql/from (sql/as :natural-earth.time-zones :n))
       (sql/join :time-zones
                 '(on (= :time-zones.natural-earth-id :n.gid))
                 :type :left)
       (sql/where '(and (is-null :time-zones.id)
                        (is-not-null :n.gid)))))
  dataset)

(defmethod insert-dataset :default [db dataset]
  (log/warnf "No insert-dataset fn defined for %s." (:table dataset))
  dataset)

;; BULK UPDATE

(defmethod update-dataset :natural-earth.airports [db dataset]
  @(sql/update db :airports
     {:country-id :u.country-id
      :region-id :u.region-id
      :gps-code :u.gps-code
      :wikipedia-url :u.wikipedia
      :location :u.geom}
     (sql/from
      (sql/as
       (sql/select db (sql/distinct
                       [(sql/as :r.country-id :country-id)
                        (sql/as :r.id :region-id)
                        (sql/as :a.gps_code :gps-code)
                        (sql/as :a.iata_code :iata-code)
                        :a.name
                        :a.wikipedia :a.geom]
                       :on [:a.iata_code])
         (sql/from (sql/as :natural-earth.airports :a))
         (sql/join (sql/as :regions :r)
                   '(on (st_intersects
                         :r.geom
                         (cast :a.geom :geography))))
         (sql/join :airports
                   '(on (= (lower :airports.iata-code)
                           (lower :a.iata_code)))))
       :u))
     (sql/where '(= :airports.iata-code :u.iata-code)))
  dataset)

(defmethod update-dataset :natural-earth.countries [db dataset]
  @(sql/update db :countries
     {:geom :u.geom
      :location '(st_centroid :u.geom)}
     (sql/from (sql/as (sql/select db [:iso_a2 :iso_a3 :iso_n3 :name :geom]
                         (sql/from :natural-earth.countries)) :u))
     (sql/where '(or (= (lower :countries.iso-3166-1-alpha-2)
                        (lower :u.iso_a2))
                     (= (lower :countries.iso-3166-1-alpha-3)
                        (lower :u.iso_a3))
                     (= (lower :countries.name)
                        (lower :u.name)))))
  dataset)

(defmethod update-dataset :natural-earth.ports [db dataset]
  @(sql/update db :ports
     {:country-id :u.country-id
      :region-id :u.region-id
      :type :u.featurecla
      :website-url :u.website
      :location :u.geom}
     (sql/from
      (sql/as
       (sql/select db (sql/distinct
                       [(sql/as :r.country-id :country-id)
                        (sql/as :r.id :region-id)
                        :p.name :p.featurecla :p.website
                        :p.geom]
                       :on [:p.name])
         (sql/from (sql/as :natural-earth.ports :p))
         (sql/join (sql/as :regions :r)
                   '(on (st_intersects
                         :r.geom
                         (cast :p.geom :geography))))
         (sql/join :ports
                   '(on (= (lower :ports.name)
                           (lower :p.name)))))
       :u))
     (sql/where '(= (lower :ports.name) (lower :u.name))))
  dataset)

(defmethod update-dataset :natural-earth.regions [db dataset]
  @(sql/update db :regions
     {:geom :u.geom
      :location '(st_centroid :u.geom)}
     (sql/from
      (sql/as
       (sql/select db [(sql/as :countries.id :country-id)
                       :n.name :n.geom]
         (sql/from (sql/as :natural-earth.regions :n))
         (sql/join :countries
                   '(on
                     (= (lower :countries.iso-3166-1-alpha-2)
                        (lower (substring :iso_3166_2
                                          from 1 for 2)))))
         (sql/join :regions
                   '(on (and (= (lower :regions.name) (lower :n.name))
                             (= :regions.country-id :countries.id)))))
       :u))
     (sql/where '(and (is-not-null :u.name)
                      (= :regions.country-id :u.country-id)
                      (= (lower :regions.name) (lower :u.name)))))
  dataset)

(defmethod update-dataset :natural-earth.time-zones [db dataset]
  @(sql/update db :time-zones
     {:geom :n.geom
      :offset :n.zone
      :places :n.places}
     (sql/from
      (sql/as
       (sql/select db [:n.geom :n.gid :n.zone
                       (sql/as '(regexp_split_to_array
                                 :n.places "\\s*,\\s*") :places)]
         (sql/from (sql/as :natural-earth.time-zones :n))
         (sql/join :time-zones
                   '(on (= :time-zones.natural-earth-id :n.gid))
                   :type :left)
         (sql/where '(and (is-null :time-zones.id)
                          (is-not-null :n.gid))))
       :n))
     (sql/where '(and (is-not-null :n.gid)
                      (= :time-zones.natural-earth-id :n.gid))))
  dataset)

(defmethod update-dataset :default [db dataset]
  (log/warnf "No update-dataset fn defined for %s." (:table dataset))
  dataset)

(defn cleanup [opts]
  (let [working-dir (:working-dir opts)]
    (FileUtils/deleteDirectory (io/file working-dir))
    (log/infof "Cleaned up working directory %s." working-dir)))

(defn run
  "Import the natural earth datasets."
  [db & [opts]]
  (import-natural-earth db opts)
  (spots/set-time-zones! db)
  (when (:cleanup opts) (cleanup opts)))

(defn -main [& args]
  (with-commandline [[opts [cmd & args]] args]
    [[c cleanup "Cleanup after import."]
     [h help "Print this help."]]
    (if (:help opts)
      (print-help "natural-earth [OPTIONS]")
      (with-db [db (config/db env)]
        (run db {:cleanup (:cleanup opts)
                 :working-dir default-working-dir})))))

(comment (time (-main)))
