(ns burningswell.config.core
  "The Burning Swell configuration."
  (:require [burningswell.config.schema :as schema]
            [#?(:clj clojure.edn :cljs cljs.reader) :as edn]
            [schema.core :as s]))

(defn- read-config
  "Read a config value from `s`."
  [s]
  (when s (edn/read-string s)))

(defn- bind-port
  "Return the bind port in `env` under `key`, or :port."
  [env key]
  (or (read-config (get env key))
      (read-config (:port env))))

(s/defn ^:always-validate api-client :- schema/ApiClient
  "Return the API client config from `env`."
  [env]
  {:scheme (keyword (:bs-api-scheme env :https))
   :server-name (:bs-api-server-name env)
   :server-port (read-config (:bs-api-server-port env))})

(s/defn ^:always-validate aws :- schema/Aws
  "Return the AWS config from `env`."
  [env]
  {:access-key (:bs-aws-access-key env)
   :account-number (:bs-aws-account-number env)
   :secret-key (:bs-aws-secret-key env)})

(s/defn ^:always-validate figwheel :- schema/Figwheel
  "Return the Figwheel config from `env`."
  [env]
  {:enable (or (read-config (:bs-figwheel-enable env)) false)})

(s/defn ^:always-validate image-proxy :- schema/ImageProxy
  "Return the Image Proxy config from `env`."
  [env]
  {:backend-url (:bs-image-proxy-backend-url env)
   :bind-address (:bs-image-proxy-bind-address env)
   :bind-port (or (bind-port env  :bs-image-proxy-bind-port) 20002)
   :scheme (keyword (:bs-image-proxy-scheme env :https))
   :server-name (:bs-image-proxy-server-name env)
   :server-port (read-config (:bs-image-proxy-server-port env))})

(s/defn ^:always-validate broker :- schema/Broker
  "Return the broker config from `env`."
  [env]
  {:password (:bs-broker-password env)
   :scheme (keyword (:bs-broker-scheme env :amqp))
   :server-name (:bs-broker-server-name env)
   :server-port (read-config (:bs-broker-server-port env))
   :username (:bs-broker-username env)
   :vhost (or (:bs-broker-vhost env) "/")})

(s/defn ^:always-validate db :- schema/Db
  "Return the database config from `env`."
  [env]
  {:name (:bs-db-name env)
   :password (:bs-db-password env)
   :test (true? (read-config (:bs-db-rollback env)))
   :scheme (keyword (:bs-db-scheme env :postgresql))
   :server-name (:bs-db-server-name env)
   :server-port (read-config (:bs-db-server-port env))
   :user (:bs-db-user env)})

(s/defn ^:always-validate facebook :- schema/Facebook
  "Return the Facebook config from `env`."
  [env]
  {:client-id (:bs-facebook-client-id env)
   :client-secret (:bs-facebook-client-secret env)
   :redirect-uri (:bs-facebook-redirect-uri env)})

(s/defn ^:always-validate flickr :- schema/Flickr
  "Return the Flickr config from `env`."
  [env]
  {:api-key (:bs-flickr-api-key env)
   :client-id (:bs-flickr-client-id env)
   :client-secret (:bs-flickr-client-secret env)})

(s/defn ^:always-validate google-maps :- schema/GoogleMaps
  "Return the Google Maps config from `env`."
  [env]
  {:api-key (:bs-google-maps-api-key env)})

(s/defn ^:always-validate google :- schema/Google
  "Return the config for Google services from `env`."
  [env]
  {:maps (google-maps env)})

(s/defn ^:always-validate reload :- schema/Reload
  "Return the namespace reload config from `env`."
  [env]
  {:disabled (true? (read-config (:bs-reload-disabled env)))
   :source-paths ["src" "test"]})

(s/defn ^:always-validate jws :- schema/JWS
  "Return the JWS config from `env`."
  [env]
  {:secret (:bs-jws-secret env)
   :expires-in (or (read-config (:bs-jws-expires-in env)) 3600)})

(s/defn ^:always-validate s3-backup :- schema/S3Backups
  "Return the S3 backup config from `env`."
  [env]
  {:bucket (:bs-s3-backup-bucket env)
   :prefix (or (:bs-s3-backup-prefix env) "backups")})

(s/defn ^:always-validate s3-photo :- schema/S3Photos
  "Return the S3 photo config from `env`."
  [env]
  {:bucket (:bs-s3-photo-bucket env)
   :prefix (or (:bs-s3-photo-prefix env) "photos")})

(s/defn ^:always-validate s3 :- schema/S3
  "Return the S3 config from `env`."
  [env]
  {:backups (s3-backup env)
   :photos (s3-photo env)})

(s/defn ^:always-validate backup :- s/Any
  "Return the backup config from `env`."
  [env]
  {:aws (aws env)
   :db (db env)
   :s3 (s3-backup env)})

(s/defn ^:always-validate web :- schema/Web
  "Return the web config from `env`."
  [env]
  {:api-client (api-client env)
   :bind-address (:bs-web-bind-address env)
   :bind-port (or (bind-port env :bs-web-bind-port) 20001)
   :google (google env)
   :facebook (facebook env)
   :figwheel (figwheel env)
   :flickr (flickr env)
   :ssl-redirect? (true? (read-config (:bs-web-ssl-redirect env)))})

(s/defn ^:always-validate web-client :- schema/WebClient
  "Return the web client config from `env`."
  [env]
  {:scheme (keyword (:bs-web-scheme env :scheme))
   :server-name (:bs-web-server-name env)
   :server-port (read-config (:bs-web-server-port env))})

(s/defn ^:always-validate api :- schema/Api
  "Return the API config from `env`."
  [env]
  {:api-client (api-client env)
   :bind-address (:bs-api-bind-address env)
   :bind-port (or (bind-port env :bs-api-bind-port) 20000)
   :broker (broker env)
   :db (db env)
   :google (google env)
   :facebook (facebook env)
   :flickr (flickr env)
   :jws (jws env)
   :reload (reload env)
   :ssl-redirect? (true? (read-config (:bs-web-ssl-redirect env)))
   :web (web-client env)})

(s/defn ^:always-validate scheduler :- schema/Scheduler
  "Return the API config from `env`."
  [env]
  {:broker (broker env)
   :db (db env)})

(s/defn ^:always-validate weather :- schema/Weather
  "Return the config for the weather service from `env`."
  [env]
  {:db (db env)
   :broker (broker env)
   :server
   {:bind-address (or (:bs-weather-bind-address env) "127.0.0.1")
    :bind-port (or (bind-port env :bs-weather-bind-port) 20002)}})

(s/defn ^:always-validate config :- schema/Config
  "Return the system config from `env`."
  [env]
  {:api (api env)
   :api-client (api-client env)
   :aws (aws env)
   :broker (broker env)
   :db (db env)
   :facebook (facebook env)
   :flickr (flickr env)
   :google (google env)
   :reload (reload env)
   :weather (weather env)
   :web (web env)
   :web-client (web-client env)})
