(ns burningswell.config.core
  "The Burning Swell configuration."
  (:require [clojure.spec.alpha :as s]
            [burningswell.config.spec]
            [#?(:clj clojure.edn :cljs cljs.reader) :as edn]))

(defn- parse
  "Read a config value from `s`."
  [s]
  (when s
    (try (edn/read-string s)
         (catch #?(:clj Exception :cljs js/Error) _ s))))

(defn- bind-port
  "Return the bind port in `env` under `key`, or :port."
  [env key]
  (or (parse (get env key))
      (parse (:port env))))

(defn valid? [spec data]
  (if (s/valid? spec data)
    true
    (throw (ex-info
            (str "Invalid configuration: " spec "\n\n"
                 (with-out-str (s/explain spec data)))
            (s/explain-data spec data)))))

(defn api-client
  "Return the API client config from `env`."
  [env]
  {:post [(valid? :burningswell.config/api-client %)]}
  {:scheme (keyword (:bs-api-scheme env "http"))
   :server-name (:bs-api-server-name env "localhost")
   :server-port (parse (:bs-api-server-port env "8001"))
   :uri "/graphql"})

(defn aws
  "Return the AWS config from `env`."
  [env]
  {:post [(valid? :burningswell.config/aws %)]}
  {:access-key (:bs-aws-access-key env)
   :account-number (:bs-aws-account-number env)
   :secret-key (:bs-aws-secret-key env)})

(defn db
  "Return the database config from `env`."
  [env]
  {:post [(valid? :burningswell.config/db %)]}
  {:name (or (:bs-db-name env)
             (:postgres-db env)
             (:pgdatabase env)
             "burningswell_development")
   :password (or (:bs-db-password env)
                 (:postgres-password env)
                 (:pgpassword env)
                 "scotch")
   :test? (true? (parse (:bs-db-test env)))
   :scheme (keyword (:bs-db-scheme env :postgresql))
   :server-name (or (:bs-db-server-name env)
                    (:pghost env)
                    "localhost")
   :server-port (parse
                 (or (:bs-db-server-port env)
                     (:pgport env)
                     "5432"))
   :username (or (:bs-db-user env)
                 (:postgres-user env)
                 (:pguser env)
                 "tiger")})

(defn facebook
  "Return the Facebook config from `env`."
  [env]
  {:post [(valid? :burningswell.config/facebook %)]}
  {:client-id (:bs-facebook-client-id env)
   :client-secret (:bs-facebook-client-secret env)
   :redirect-uri (:bs-facebook-redirect-uri
                  env "http://localhost:8001/oauth/facebook/login")})

(defn flickr
  "Return the Flickr config from `env`."
  [env]
  {:post [(valid? :burningswell.config/flickr %)]}
  {:api-key (:bs-flickr-api-key env)
   :client-id (:bs-flickr-client-id env)
   :client-secret (:bs-flickr-client-secret env)})

(defn google-analytics
  "Return the Google Analytics config from `env`."
  [env]
  {:post [(valid? :burningswell.config.google/analytics %)]}
  {:tracking-id (:bs-google-analytics-tracking-id env)})

(defn google-maps
  "Return the Google Maps config from `env`."
  [env]
  {:post [(valid? :burningswell.config.google/maps %)]}
  {:api-key (:bs-google-maps-api-key env)})

(defn google-storage
  "Return the Google Maps config from `env`."
  [env]
  {:post [(valid? :burningswell.config.google/storage %)]}
  {:bucket (:bs-google-storage-bucket env "burningswell-development")})

(defn google
  "Return the config for Google services from `env`."
  [env]
  {:post [(valid? :burningswell.config/google %)]}
  {:analytics (google-analytics env)
   :client-id (:bs-google-client-id env)
   :client-secret (:bs-google-client-secret env)
   :maps (google-maps env)
   :storage (google-storage env)
   :redirect-uri (:bs-google-redirect-uri
                  env "http://localhost:8001/oauth/google/login")})

(defn kafka
  "Return the Kafka config from `env`."
  [env]
  {:post [(valid? :burningswell.config/kafka %)]}
  {:bootstrap.servers (:bs-kafka-bootstrap-servers env "localhost:9092")})

(defn linkedin
  "Return the Linkedin config from `env`."
  [env]
  {:post [(valid? :burningswell.config/linkedin %)]}
  {:client-id (:bs-linkedin-client-id env)
   :client-secret (:bs-linkedin-client-secret env)})

(defn mapbox
  "Return the Mapbox config from `env`."
  [env]
  {:post [(valid? :burningswell.config/mapbox %)]}
  {:api-key (:bs-mapbox-api-key env)
   :style (:bs-mapbox-style env "mapbox://styles/mapbox/light-v8")})

(defn nrepl
  "Return the NREPL config from `env`."
  [env]
  {:post [(valid? :burningswell.config/nrepl %)]}
  {:bind-address (:bs-nrepl-bind-address env "0.0.0.0")
   :bind-port (parse (:bs-nrepl-bind-port env))})

(defn jwt
  "Return the JWT config from `env`."
  [env]
  {:post [(valid? :burningswell.config/jwt %)]}
  {:secret (:bs-jwt-secret env)
   :expires-in (parse (:bs-jwt-expires-in env "3600"))})

(defn schema-registry
  "Return the schema registry config from `env`."
  [env]
  {:post [(valid? :burningswell.config/schema-registry %)]}
  {:url (or (:bs-schema-registry-url env) "http://localhost:8081")
   :max-capacity (parse (or (::bs-schema-registry-max-capacity env) "100"))})

(defn s3-backup
  "Return the S3 backup config from `env`."
  [env]
  {:post [(valid? :burningswell.config.s3/backups %)]}
  {:bucket (:bs-s3-backup-bucket env "burningswell-dev")
   :prefix (:bs-s3-backup-prefix env "backups")})

(defn s3-photo
  "Return the S3 photo config from `env`."
  [env]
  {:post [(valid? :burningswell.config.s3/photos %)]}
  {:bucket (:bs-s3-photo-bucket env "burningswell-dev")
   :prefix (:bs-s3-photo-prefix env "photos")})

(defn s3
  "Return the S3 config from `env`."
  [env]
  {:post [(valid? :burningswell.config/s3 %)]}
  {:backups (s3-backup env)
   :photos (s3-photo env)})

(defn twitter
  "Return the Twitter config from `env`."
  [env]
  {:post [(valid? :burningswell.config/twitter %)]}
  {:client-id (:bs-twitter-client-id env)
   :client-secret (:bs-twitter-client-secret env)})

(defn backup
  "Return the backup config from `env`."
  [env]
  {:post [(valid? :burningswell.config/backup %)]}
  {:aws (aws env)
   :db (db env)
   :s3 (s3-backup env)})

(defn web-client
  "Return the web client config from `env`."
  [env]
  {:post [(valid? :burningswell.config/web-client %)]}
  {:scheme (keyword (:bs-web-scheme env "http"))
   :server-name (:bs-web-server-name env "localhost")
   :server-port (parse (:bs-web-server-port env "8000"))})

(defn photos
  "Return the config for the photos service from `env`."
  [env]
  {:post [(valid? :burningswell.config/photos %)]}
  {:bind-address (:bs-photos-bind-address env "0.0.0.0")
   :bind-port (parse (:bs-photos-bind-port env "8004"))
   :scheme (keyword (:bs-photos-scheme env "http"))
   :server-name (:bs-photos-server-name env "localhost")
   :server-port (parse (:bs-photos-server-port env "8004"))
   :version (:bs-photos-version env "latest")})

(defn smtp
  "Return the SMTP config from `env`."
  [env]
  {:post [(valid? :burningswell.config/smtp %)]}
  {:scheme (keyword (:bs-smtp-scheme env "smtps"))
   :server-name (:bs-smtp-server-name env "smtp.gmail.com")
   :server-port (parse (:bs-smtp-server-port env "465"))
   :username (:bs-smtp-username env)
   :password (:bs-smtp-password env)})

(defn web
  "Return the web config from `env`."
  [env]
  {:post [(valid? :burningswell.config/web %)]}
  {:api-client (api-client env)
   :async-js (true? (parse (:bs-web-async-js env)))
   :bind-address (:bs-web-bind-address env "0.0.0.0")
   :bind-port (parse (:bs-web-bind-port env "8000"))
   :facebook (facebook env)
   :flickr (flickr env)
   :google (google env)
   :mapbox (mapbox env)
   :photos (photos env)
   :web-client (web-client env)})

(defn api
  "Return the API config from `env`."
  [env]
  {:post [(valid? :burningswell.config/api %)]}
  {:api-client (api-client env)
   :bind-address (:bs-api-bind-address env "127.0.0.0")
   :bind-port (parse (:bs-api-bind-port env "8001"))
   :db (db env)
   :facebook (facebook env)
   :flickr (flickr env)
   :google (google env)
   :jwt (jwt env)
   :kafka (kafka env)
   :linkedin (linkedin env)
   :nrepl (nrepl env)
   :photos (photos env)
   :twitter (twitter env)
   :web (web-client env)})

(defn worker
  "Return the worker config from `env`."
  [env]
  {:bind-address (:bs-worker-bind-address env "0.0.0.0")
   :bind-port (parse (:bs-worker-bind-port env "8003"))
   :nrepl (nrepl env)})

(defn scheduler
  "Return the API config from `env`."
  [env]
  {:db (db env)})

(defn config
  "Return the system config from `env`."
  [env]
  {:post [(valid? :burningswell/config %)]}
  {:api (api env)
   :api-client (api-client env)
   :aws (aws env)
   :db (db env)
   :facebook (facebook env)
   :flickr (flickr env)
   :google (google env)
   :kafka (kafka env)
   :nrepl (nrepl env)
   :photos (photos env)
   :smtp (smtp env)
   :web (web env)
   :web-client (web-client env)
   :worker (worker env)})
