(ns burningswell.flickr
  (:require [clj-http.client :as http]
            [geo.postgis]
            [clojure.spec.alpha :as s]))

(s/def ::width pos-int?)
(s/def ::height pos-int?)

(s/def ::label
  #{"Large" "Medium" "Original" "Small" "Square" "Thumbnail"})

(s/def ::size (s/keys :req-un [::width ::height ::label]))

(def licenses
  "The Flickr license types."
  [{:id 0
    :name "All Rights Reserved"
    :usable? false}
   {:id 1
    :name "Attribution-NonCommercial-ShareAlike License"
    :url "http://creativecommons.org/licenses/by-nc-sa/2.0/"
    :usable? false}
   {:id 2
    :name "Attribution-NonCommercial License"
    :url "http://creativecommons.org/licenses/by-nc/2.0/"
    :usable? false}
   {:id 3
    :name "Attribution-NonCommercial-NoDerivs License"
    :url "http://creativecommons.org/licenses/by-nc-nd/2.0/"
    :usable? false}
   {:id 4
    :name "Attribution License"
    :url "http://creativecommons.org/licenses/by/2.0/"
    :usable? true}
   {:id 5
    :name "Attribution-ShareAlike License"
    :url "http://creativecommons.org/licenses/by-sa/2.0/"
    :usable? true}
   {:id 6
    :name "Attribution-NoDerivs License"
    :url "http://creativecommons.org/licenses/by-nd/2.0/"
    :usable? true}
   {:id 7
    :name "No known copyright restrictions"
    :url "http://flickr.com/commons/usage/"
    :usable? true}
   {:id 8
    :name "United States Government Work"
    :url "http://www.usa.gov/copyright.shtml"
    :usable? true}])

(defn client
  "Returns a Flickr client using `api-key`."
  [{:keys [api-key client-id client-secret]}]
  (fn [request]
    (-> (assoc-in request [:query-params :api_key] api-key)
        (http/request))))

(defn photo-url
  "Returns the Flickr URL of `photo`."
  [photo & [size]]
  (str "http://farm" (:farm photo) ".staticflickr.com/"
       (:server photo) "/" (:id photo) "_" (:secret photo)
       (if size (str "_" (name size)))
       ".jpg"))

(defn request
  "Build a Flickr request."
  [method params]
  {:url "https://api.flickr.com/services/rest"
   :as :auto
   :method :get
   :accept "application/json"
   :query-params
   (assoc params
          :method (name method)
          :format "json"
          :nojsoncallback "1")})

(defn search
  "Request a list of photos matching some criteria."
  [flickr-client opts]
  (-> (request :flickr.photos.search opts)
      (flickr-client) :body :photos :photo))

(defn sizes
  "Request a image sizes of `photo-id`."
  [flickr-client photo-id & [opts]]
  (-> (request :flickr.photos.getSizes (assoc opts :photo_id photo-id))
      (flickr-client) :body :sizes :size))

(defn user-info
  "Return information about a user."
  [flickr-client user-id & [opts]]
  (-> (request :flickr.people.getInfo (assoc opts :user_id user-id))
      (flickr-client) :body :person))

(defn photo-info
  "Return information about a photo."
  [flickr-client photo & [opts]]
  (-> (request :flickr.photos.getInfo (assoc opts :photo_id photo))
      (flickr-client) :body :photo))
