(ns burningswell.file
  (:import [java.nio.file Files Path]
           [org.apache.commons.codec.binary Base64])
  (:require [clojure.edn :as edn]
            [clojure.java.io :refer [delete-file file]]
            [no.en.core :refer [base64-encode]]))

(defn edn-line-seq
  "Parse the lines in `reader` as EDN and return them as a seq."
  [reader]
  (->> (line-seq reader)
       (map #(edn/read-string {:readers *data-readers*} %1))))

(defn delete-files
  "Delete all filenames."
  [filenames & [silently]]
  (doseq [filename filenames]
    (delete-file (file filename) silently)))

(defn file-exists?
  "Returns true if `filename` exists, otherwise false."
  [filename] (.exists (file filename)))

(defmacro with-tmp-files
  "Evaluate `body` with `tmp-syms` to temporary file names, that are
  delete after `body` has been evaluated."
  [[& tmp-syms] & body]
  (let [tmp-paths (mapcat
                   #(vector %1 `(str "/tmp/" (java.util.UUID/randomUUID)))
                   tmp-syms)]
    `(let [~@tmp-paths]
       (try
         ~@body
         (finally
           (delete-files [~@tmp-syms] true))))))

(defn read-all-bytes
  "Read all bytes from `filename`."
  [filename]
  (-> (.toPath (file filename))
      (Files/readAllBytes)))

(defn read-base64
  "Read the file `filename` and return as a Base64 encoded string."
  [filename]
  (-> (read-all-bytes filename)
      (Base64/encodeBase64)
      (String.)))
