(ns burningswell.namespace
  (:import clojure.lang.Var)
  (:require [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [ns-tracker.core :as tracker]))

(defn immigrate
  "Create a public var in this namespace for each public var in the
  namespaces named by ns-names. The created vars have the same name, root
  binding, and metadata as the original except that their :ns metadata
  value is this namespace."
  [& ns-names]
  (doseq [ns ns-names]
    (require ns)
    (doseq [[sym var] (ns-publics ns)]
      (let [v (if (.hasRoot ^Var var)
                (var-get var))
            var-obj (if v (intern *ns* sym v))]
        (when var-obj
          (alter-meta! var-obj
                       (fn [old] (merge (meta var) old)))
          var-obj)))))

(defn refer-public
  [ns & symbols]
  (let [symbols (set symbols)]
    (require ns)
    (doseq [[sym var] (ns-publics ns)
            :let [sym (with-meta sym (assoc (meta var) :ns *ns*))]
            :when (contains? symbols sym)]
      (if (.hasRoot ^Var var)
        (intern *ns* sym var)
        (intern *ns* sym)))))

(defn refer-private [ns & symbols]
  (let [symbols (set symbols)]
    (doseq [[symbol var] (ns-interns ns) :when (:private (meta var))]
      (when (or (empty? symbols) (contains? symbols symbol))
        (intern *ns* symbol var)))))

(defrecord Reload [reload source-paths]
  component/Lifecycle
  (start [component]
    (if (and reload (not (empty? source-paths)))
      (let [track (tracker/ns-tracker source-paths)
            done (atom false)]
        (doto (Thread.
               (fn []
                 (while (not @done)
                   (try (doseq [ns-sym (track)]
                          (require ns-sym :reload))
                        (catch Throwable e (.printStackTrace e)))
                   (Thread/sleep 500))))
          (.setDaemon true)
          (.start))
        (log/infof "Namespace reloader started.")
        (assoc component :stop (fn [] (swap! done not))))
      component))
  (stop [component]
    (when-let [stop (:stop component)]
      (stop)
      (log/infof "Namespace reloader stopped."))
    (dissoc component :stop)))

(defn new-reload [config]
  (->Reload (:enabled config) (:source-paths config)))
