(ns burningswell.routes
  (:require [bidi.bidi :as bidi]
            [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as gen]
            [clojure.string :as str]
            [no.en.core :refer [format-url parse-url]]))

(s/def ::request-method
  #{:get :delete :post :put})

(s/def ::query-param-name
  (s/with-gen (s/and simple-keyword? #(seq (name %)))
    #(gen/fmap keyword (gen/string-alphanumeric))))

(s/def ::query-param-value
  (s/with-gen (s/and (s/and string? #(seq (name %)))
                     #(nil? (str/index-of % "?"))
                     #(nil? (str/index-of % "=")))
    #(gen/string-alphanumeric)))

(s/def ::query-params
  (s/map-of ::query-param-name ::query-param-value))

(def default-gen-opts
  "The default generator options."
  {:params {:default {:id (s/gen pos-int?)}}})

(def ^:private params
  {:name [#".*" :name]
   :id [#"\d+" :id]})

(def ^:private slug-id-name
  ["/" (:id params) "-" (:name params)])

(def routes
  "The Burning Swell web routes."
  ["/"
   {"" :home

    "airports"
    {"" :airports
     "/nearby" :airports-nearby
     slug-id-name :airport}

    "continents"
    {"" :continents
     "/nearby" :continents-nearby
     slug-id-name :continent}

    "countries"
    {"" :countries
     "/nearby" :countries-nearby
     slug-id-name :country}

    "graphql" :graphql

    "ports"
    {"" :ports
     "/nearby" :ports-nearby
     slug-id-name :port}

    "regions"
    {"" :regions
     "/nearby" :regions-nearby
     slug-id-name :region}

    "oauth"
    {"" :oauth
     ["/" (:name params) "/callback"] :oauth-callback
     ["/" (:name params) "/connect"] :oauth-connect}

    "state" :state
    "settings" :settings
    "signin" :signin
    "signout" :signout
    "signup" :signup

    "spots"
    {"" :spots
     "/nearby" :spots-nearby
     slug-id-name :spot}

    "users"
    {"" :users
     "/nearby" :users-nearby
     slug-id-name :user}

    "weather/models"
    {"" :weather-models
     slug-id-name :weather-model}

    "weather/variables"
    {"" :weather-variables
     slug-id-name :weather-variable}}])

(defn match
  "Match the `path` against `routes`."
  [path & [opts]]
  (when path
    (let [opts' (apply concat (seq opts))]
      (some-> (apply bidi/match-route routes path opts')
              (assoc :uri path)))))

(defn match-url
  "Match the `url` against `routes`."
  [url & [opts]]
  (match (:uri (parse-url url)) opts))

(defn path [handler & [opts]]
  (apply bidi/path-for routes handler (apply concat (seq opts))))

(defn url-spec [client handler & [opts]]
  (->> (path handler opts)
       (assoc client :uri)))

(defn url-str [client handler & [opts]]
  (format-url (url-spec client handler opts)))

;; Generate requests

(defn- collect-params [xs]
  (some->> xs (map (fn [x]
                     (cond
                       (keyword? x)
                       {:name x}
                       (vector? x)
                       {:name (last x)
                        :pattern (first x)}
                       :else nil)))
           (remove nil?)
           (not-empty)
           (vec)))

(defn- flatten-routes [routes]
  (->> (for [[k v] routes]
         (cond
           (keyword? v)
           (let [params (collect-params k)]
             [(cond-> {:handler v}
                params (assoc :params params))])
           (and (string? k)
                (map? v))
           (flatten-routes v)
           :else nil))
       (apply concat)))

(defn- collect-routes [routes]
  (let [routes (flatten-routes routes)]
    (zipmap (map :handler routes) routes)))

(defn- gen-request-params [handler params & [opts]]
  (->> (for [{:keys [name pattern]} params]
         [name (or (get-in opts [:params handler name])
                   (get-in opts [:params :default name])
                   (gen/string-alphanumeric))])
       (apply concat)
       (apply gen/hash-map)))

(defn- gen-request-uri [handler params-gen]
  (gen/fmap #(path handler %) params-gen))

(defn- gen-routes
  "Returns a generator that produces Ring request maps for the `routes`."
  [routes & [opts]]
  (let [except (some-> opts :except set)
        only (some-> opts :only set)
        routes (collect-routes (second routes))]
    (->> (for [[handler {:keys [params]}] routes
               :when (and (or (empty? only)
                              (contains? only handler))
                          (or (empty? except)
                              (not (contains? except handler))))]
           (gen/bind (gen-request-params handler params opts)
                     #(gen/hash-map
                       :handler (gen/return handler)
                       :request-method (s/gen ::request-method)
                       :route-params (gen/return %1)
                       :scheme (s/gen #{:https :http})
                       :server-name (gen/return "www.burningswell.com")
                       :uri (gen-request-uri handler (gen/return %1))
                       :query-params (if (= (:query-params? opts) false)
                                       (gen/return nil)
                                       (s/gen ::query-params)))))
         (gen/one-of))))

(defn gen-request
  "Returns a generator that produces Ring request maps."
  [& [opts]]
  (gen-routes routes (merge default-gen-opts opts)))

(defn gen-url
  "Returns a generator that produces urls matching `routes`."
  [& [opts]]
  (gen/fmap format-url (gen-request opts)))

(defn gen-path
  "Returns a generator that produces paths matching `routes`."
  [& [opts]]
  (gen/fmap :uri (gen-request opts)))

(s/def ::path
  (s/with-gen (s/and string? #(match %)) #(gen-path)))

(s/def ::url
  (s/with-gen (s/and string? #(match-url %)) #(gen-url)))

(s/def ::request
  (s/with-gen (s/and map? #(match (:uri  %))) #(gen-request)))
