(ns burningswell.db.photos2
  (:require [datumbazo.core :as sql :refer [column]]
            [datumbazo.table :refer [deftable]]
            [datumbazo.record :refer [select-class]]
            [clojure.spec :as s]
            [clojure.spec.gen :as gen]))

(deftable photos
  "The photos database table."
  (column :id :serial :primary-key? true)
  (column :user-id :integer :not-null? true :references :users/id)
  (column :flickr-id :bigint :unique? true)
  (column :flickr-owner-id :text)
  (column :flickr-owner-name :text)
  (column :flickr-owner-url :text)
  (column :status :text)
  (column :title :text)
  (column :url :text)
  (column :likes :integer)
  (column :dislikes :integer)
  (column :created-at :timestamp :not-null? true)
  (column :updated-at :timestamp :not-null? true))

(defmethod select-class Photo
  [db class & [opts]]
  (sql/select db [:photos.id
                  (sql/as '(cast :photos.location :geometry) :location)
                  :photos.url
                  :photos.status
                  :photos.title
                  :photos.likes
                  :photos.dislikes
                  :photo-likes.like
                  :photos.created-at
                  :photos.updated-at
                  (sql/as `(json_build_object
                            "id" :photos.flickr-id
                            "owner"
                            (case (and (is-null :photos.flickr-owner-id)
                                       (is-null :photos.flickr-owner-name)
                                       (is-null :photos.flickr-owner-url)) nil
                                  (json_build_object
                                   "id" :photos.flickr-owner-id
                                   "name" :photos.flickr-owner-name
                                   "url" :photos.flickr-owner-url)))
                          :flickr)
                  (sql/as `(json_build_object
                            "user" (json-embed-user :users))
                          :_embedded)]
    (sql/from :photos)
    (sql/join :users '(on (= :users.id :photos.user-id)) :type :left)
    (sql/join :photo-likes
              `(on (and (= :photo-likes.photo-id :photos.id)
                        (= :photo-likes.user-id ~(-> opts :user :id))))
              :type :left)
    ;; (fulltext (:query opts) :photos.title)
    ;; (order-by (or (:order-by opts)
    ;;               (desc :photos.created-at)))
    ;; (paginate (:page opts) (:per-page opts))
    ))

(gen/sample (s/gen ::photos))
