(ns burningswell.db.airports
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(defn- select-all
  "Returns all airports."
  [db & [opts]]
  (let [{:keys [query page per-page]} opts]
    (select db [:airports.id
                :airports.name
                (as '(cast :airports.location :geometry) :location)
                :airports.gps-code
                :airports.iata-code
                :airports.wikipedia-url
                :airports.created-at
                :airports.updated-at
                (as `(json_build_object
                      "country" (json-embed-country :countries)
                      "region" (json-embed-region :regions))
                    :_embedded)]
      (from :airports)
      (join :countries.id :airports.country-id)
      (join :regions.id :airports.region-id :type :left)
      (fulltext query :airports.name)
      (if-let [location (:location opts)]
        (order-by-distance :airports.location location)
        (order-by :airports.name))
      (paginate page per-page))))

(s/defn all :- [Airport]
  "Return all airports in `db`."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn by-id :- (s/maybe Airport)
  "Return the country in `db` by `id`."
  [db :- Database id :- s/Num]
  (first @(compose
           (select-all db)
           (where `(= :airports.id (cast ~id :integer))))))

(s/defn by-iata-code :- (s/maybe Airport)
  "Return the country in `db` by `iata-code`."
  [db :- Database iata-code :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :airports.iata-code ~iata-code)))))

(s/defn by-name :- (s/maybe Airport)
  "Return the country in `db` by `name`."
  [db :- Database name :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :airports.name ~name)))))

(defn- row [continent]
  (assoc (select-keys continent [:name :gps-code :iata-code
                                 :location :wikipedia-url])
         :country-id (-> continent :_embedded :country :id)
         :region-id (-> continent :_embedded :region :id)))

(s/defn delete
  "Delete `airport` from `db`."
  [db :- Database airport :- Airport]
  (->> @(sql/delete db :airports
          (where `(= :airports.id
                     ~(:id airport))))
       first :count))

(s/defn in-continent :- [Airport]
  "Returns all airports in `continent`."
  [db :- Database continent :- Continent & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :countries.continent-id ~(:id continent)))))

(s/defn in-country :- [Airport]
  "Returns all airports in `country`."
  [db :- Database country :- Country & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :airports.country-id ~(:id country)))))

(s/defn in-region :- [Airport]
  "Returns all airports in `region`."
  [db :- Database region :- Region & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :airports.region-id ~(:id region)))))

(s/defn insert
  "Insert `airport` into `db`."
  [db :- Database airport]
  (->> @(sql/insert db :airports []
          (values [(row airport)])
          (returning :id))
       first :id (by-id db)))

(s/defn update
  "Update `airport` in `db`."
  [db :- Database airport]
  (->> @(sql/update db :airports
          (row airport)
          (where `(= :airports.id
                     ~(:id airport)))
          (returning :id))
       first :id (by-id db)))

(s/defn honolulu :- (s/maybe Airport)
  "Returns the Honolulu airport."
  [db :- Database]
  (by-iata-code db "HNL"))
