(ns burningswell.db.regions
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.util :as util :refer :all]
            [burningswell.db.schemas :refer :all]
            [burningswell.db.photos :as photos]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [clojure.java.jdbc :as jdbc]
            [geo.postgis :as geo]
            [no.en.core :refer [parse-integer]]
            [schema.core :as s])
  (:import java.awt.Color
           sqlingvo.db.Database
           org.postgis.Point))

(defn- select-all [db & [opts]]
  (select db [:regions.id
              :regions.name
              (as '(cast :regions.location :geometry) :location)
              :regions.airport-count
              :regions.port-count
              :regions.spot-count
              :regions.user-count
              :regions.created-at
              :regions.updated-at
              (as `(json_build_object
                    "country" (json-embed-country :countries)
                    "photo" ~embedded-photo)
                  :_embedded)]
    (from :regions)
    (join :countries.id :regions.country-id)
    (join :photos.id :regions.photo-id :type :left)
    (fulltext (:query opts) :regions.name)
    (within-bounding-box :regions.geom (:bounding-box opts))
    (if-let [min-spots (:min-spots opts)]
      (where `(>= :regions.spot-count ~min-spots) :and))
    (if-let [location (:location opts)]
      (order-by-distance :regions.geom location)
      (order-by :regions.name))
    (paginate (:page opts) (:per-page opts))))

(s/defn country-id :- s/Int
  "Return the country id of `region`."
  [region]
  (-> region :_embedded :country :id))

(s/defn all :- [Region]
  "Return all regions in `db`."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn by-country :- [Region]
  "Return the regions of `country` in `db`."
  [db :- Database country :- Country & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :regions.country-id ~(:id country)))))

(s/defn by-id :- (s/maybe Region)
  "Return the region in `db` by `id`."
  [db :- Database id :- s/Num]
  (first @(compose
           (select-all db)
           (where `(= :regions.id ~id)))))

(s/defn by-ids :- [Region]
  "Return all regions in `db` by the list of `ids`."
  [db :- Database ids :- [s/Num] & [opts]]
  @(compose
    (select-all db opts)
    (where `(in :regions.id ~ids))))

(s/defn by-name :- (s/maybe Region)
  "Return the country in `db` by `name`."
  [db :- Database name :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :regions.name ~name)))))

(s/defn by-location :- [Region]
  "Return the country in `db` by `location`."
  [db :- Database location :- Point]
  (let [location (geo/geometry location)]
    @(compose
      (select-all db)
      (where `(st_intersects :regions.geom (cast ~location :geography)))
      (order-by `(st_distance (cast ~location :geography) :regions.geom)))))

(s/defn within-distance :- [Region]
  "Return the regions in `db` within `distance` in km to `location`."
  [db :- Database location :- Point distance :- s/Num & [opts]]
  @(compose
    (select-all db (assoc opts :location location))
    (within-distance-to :regions.geom location distance)))

(s/defn closest :- (s/maybe Region)
  "Find the closest region to `location`."
  [db :- Database location :- Point & [distance :- s/Num]]
  (or (first (within-distance db location (or distance 20)))
      (first (by-location db location))))

(s/defn by-spot :- (s/maybe Region)
  "Load the region of `spot` from `db`."
  [db :- Database spot :- Spot]
  (some->> spot :_embedded :region :id
           (by-id db)))

(defn- row [continent]
  (assoc (select-keys continent [:name :location])
         :country-id (-> continent :_embedded :country :id)))

(s/defn delete
  "Delete `region` from `db`."
  [db :- Database region :- Region]
  (->> @(sql/delete db :regions
          (where `(= :regions.id
                     ~(:id region))))
       first :count))

(s/defn in-continent :- [Region]
  "Return all regions in `continent`."
  [db :- Database continent :- Continent & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :countries.continent-id ~(:id continent)))))

(s/defn in-country :- [Region]
  "Return all regions in `country`."
  [db :- Database country :- Country & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :regions.country-id ~(:id country)) :and)))

(s/defn insert :- Region
  "Insert `region` into `db`."
  [db :- Database region]
  (->> @(sql/insert db :regions []
          (values (row region))
          (returning :id))
       first :id (by-id db)))

(s/defn update :- (s/maybe Region)
  "Update `region` in `db`."
  [db :- Database region]
  (->> @(sql/update db :regions (row region)
                    (where `(= :regions.id
                               ~(:id region)))
                    (returning :id))
       first :id (by-id db)))

(s/defn andalucia :- (s/maybe Region)
  "Return the region Andalucía, Spain."
  [db :- Database]
  (by-name db "Andalucía"))

(s/defn bali :- (s/maybe Region)
  "Return the region Bali, Indonesia."
  [db :- Database]
  (by-name db "Bali"))

(s/defn hawaii :- (s/maybe Region)
  "Return the region Hawaii, United States."
  [db :- Database]
  (by-name db "Hawaii"))

(s/defn pais-vasco :- (s/maybe Region)
  "Return the region País Vasco, Spain."
  [db :- Database]
  (by-name db "País Vasco"))

(s/defn update-counters
  "Update the counters of all regions."
  [db :- Database]
  (->> [(select db ['(update-region-airport-count)])
        (select db ['(update-region-port-count)])
        (select db ['(update-region-spot-count)])
        (select db ['(update-region-user-count)])]
       (map (comp first deref))
       (apply merge)))

(s/defn top-photos
  "Return the top photo for each region."
  [db :- Database]
  (select db (distinct
              [:photos-regions.region-id
               :photos-regions.photo-id]
              :on [:photos-regions.region-id] )
    (from :photos-regions)
    (join :photos.id :photos-regions.photo-id)
    (order-by :photos-regions.region-id
              (desc :photos.created-at))))

(s/defn update-photos
  "Update the photos of all regions."
  [db :- Database]
  (sql/update db :regions
    {:photo-id :top-photos.photo-id}
    (from (as (top-photos db) :top-photos))
    (where `(= :regions.id :top-photos.region-id))))

(s/defn assoc-photo :- [Region]
  "Assoc the photo and their images to all `regions`."
  [db :- Database regions :- [Region]]
  (let [photos (zip-by-id (photos/by-regions db regions {:images true}))]
    (map #(->> (get photos (-> % :_embedded :photo :id))
               (assoc-in % [:_embedded :photo]))
         regions)))

(s/defn as-svg :- s/Any
  "Return the `region` as SVG."
  [db :- Database region :- Region]
  (->> @(select db [(as '(st_assvg (cast :geom :geometry)) :svg)]
          (from :regions)
          (where `(= :regions.id ~(:id region))))
       first :svg))

(s/defn as-png :- s/Any
  "Return the `region` as PNG."
  [db :- Database region :- Region & [opts]]
  (let [{:keys [width height color]} opts]
    (->> @(select db [(as (util/geometry-as-png width height color) :png)]
            (from :regions)
            (where `(= :regions.id ~(:id region))))
         first :png)))

(comment

  (def my-db (-> reloaded.repl/system :db))

  @(burningswell.db.countries/update-photos my-db)
  @(burningswell.db.regions/update-photos my-db)

  )
