(ns burningswell.db.backup
  (:require [aws.sdk.s3 :as s3]
            [burningswell.config.core :as config]
            [burningswell.db.connection :as db]
            [burningswell.db.materialized-views :as materialized-views]
            [burningswell.system :refer [defsystem]]
            [clj-time.coerce :refer [to-date-time]]
            [clj-time.core :refer [interval in-minutes now]]
            [clj-time.format :refer [formatters parse unparse]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [commandline.core :refer [print-help with-commandline]]
            [com.stuartsierra.component :as component]
            [datumbazo.shell :as shell]
            [environ.core :refer [env]])
  (:import [org.apache.commons.io FileUtils]))

(def formatter
  "The date formatter for backups."
  (:date-hour-minute-second formatters))

(defsystem backup-system
  "The backup system."
  [{:keys [aws db s3] :as config}]
  (component/system-map
   :aws aws
   :db (db/new-db db)
   :s3 s3))

(defn mega-bytes
  "Return `bytes` in megabytes."
  [bytes]
  (long (/ bytes 1024 1024)))

(defn timestamp
  "Return the formatted `time`."
  [time]
  (unparse formatter time))

(defn backup-path
  "Return the backup path."
  [system & paths]
  (str (apply io/file "backups" paths)))

(defn s3-path
  "Return the S3 path."
  [system & paths]
  (str (-> system :s3 :prefix) "/" (str/join "/" paths)))

(defn- db-file
  "Return db filename for `type` and `time`."
  [db type time]
  (str (:name db) "/" (timestamp time) "/" (name type) ".sql"))

(defn data-file
  "Return the data filename for `time`."
  [db time]
  (str (db-file db :data time) ".gz"))

(defn schema-file
  "Return the schema filename for `time`."
  [db time]
  (db-file db :schema time))

(defn make-backup
  "Make a new backup record for `time`. "
  [{:keys [db] :as system} time]
  {:db (:name db)
   :local-data (backup-path system (data-file db time))
   :local-schema (backup-path system (schema-file db time))
   :s3-data (s3-path system (data-file db time))
   :s3-schema (s3-path system (schema-file db time))
   :time time})

;; Create backup

(defn dump-data
  "Dump the database data for `backup`."
  [{:keys [db]} backup]
  (io/make-parents (io/file (:local-data backup)))
  (shell/exec-checked-script
   "Dumping data"
   ("export" ~(format "PGPASSWORD=\"%s\"" (:password db)))
   ("pg_dump"
    "--dbname" ~(:db backup)
    "--host" ~(:server-name db)
    "--port" ~(:server-port db)
    "--username" ~(:user db)
    "--data-only"
    "--exclude-table" "weather.rasters"
    "|" "gzip" ">" ~(:local-data backup)))
  (log/infof "Dumped the data for database %s." (:name db))
  backup)

(defn dump-schema
  "Dump the database schema for `backup`."
  [{:keys [db]} backup]
  (io/make-parents (io/file (:local-schema backup)))
  (shell/exec-checked-script
   "Dumping schema"
   ("export" ~(format "PGPASSWORD=\"%s\"" (:password db)))
   ("pg_dump"
    "--dbname" ~(:db backup)
    "--host" ~(:server-name db)
    "--port" ~(:server-port db)
    "--username" ~(:user db)
    "--schema-only"
    ">" ~(:local-schema backup)))
  (log/infof "Dumped the schema for database %s." (:name db))
  backup)

(defn copy-local-to-s3
  "Copy `source` from the local filesystem to to `target` on S3."
  [system source target]
  (s3/put-object (:aws system) (-> system :s3 :bucket) target (io/file source)))

(defn copy-s3-to-local
  "Copy `source` from S3 to `target` on the local filesystem."
  [system source target]
  (let [output (io/file target)
        bucket (-> system :s3 :bucket)
        object (s3/get-object (:aws system) bucket source)]
    (io/make-parents output)
    (io/copy (:content object) output)))

(defn upload-data
  "Save the data to S3."
  [system backup]
  (copy-local-to-s3 system (:local-data backup) (:s3-data backup) )
  (log/infof "Saved the data for database %s to S3." (:db backup))
  backup)

(defn upload-schema
  "Save the schema to S3."
  [system backup]
  (copy-local-to-s3 system (:local-schema backup) (:s3-schema backup) )
  (log/infof "Saved the schema for database %s to S3." (:db backup))
  backup)

(defn create-backup
  "Create a database backup at `time`."
  [config time & [opts]]
  (let [system (backup-system config)]
    (let [started-at (now)
          backup (->> (make-backup system time)
                      (dump-schema system)
                      (upload-schema system)
                      (dump-data system)
                      (upload-data system))]
      (when (:cleanup opts)
        (let [dir (.getParentFile (io/file (:local-schema backup)))]
          (FileUtils/deleteDirectory dir)))
      (log/infof "Backup for database %s finished in %s minutes."
                 (:db backup) (in-minutes (interval started-at (now))))
      backup)))

;; List backups

(def s3-data-regex
  "Regular expression matching backup data on S3."
  #"(.*/)?([^/]+)/([^/]+)/data.sql.gz")

(defn parse-s3-data-key
  "Parse the S3 key."
  [s]
  (when-let [matches (re-matches s3-data-regex (str s))]
    (when-let [time (parse formatter (nth matches 3))]
      {:db (nth matches 2)
       :time time})))

(defn parse-s3-backups
  "Parse the S3 backups from `results`."
  [system results]
  (for [object (:objects results)
        :let [backup (parse-s3-data-key (:key object))]
        :when backup]
    (-> (make-backup system (:time backup))
        (assoc :metadata (:metadata object)))))

(defn print-s3-backups
  "Parse the S3 backups from `results`."
  [backups]
  (doseq [[db backups] (group-by :db backups)]
    (log/infof "Backups available for database %s:" db)
    (doseq [{:keys [time metadata]} backups]
      (log/infof "- Time: %s, Size: %s MB" time
                 (mega-bytes (:content-length metadata)))))
  backups)

(defn find-s3-backups
  "Find all backups on S3."
  [system]
  (->> (s3/list-objects
        (:aws system)
        (-> system :s3 :bucket)
        {:prefix (-> system :s3 :prefix)})
       (parse-s3-backups system)))

(defn list-backups
  "List all database backups on S3."
  [config]
  (let [system (backup-system config)]
    (->> (find-s3-backups system)
         (print-s3-backups))))

;; Restore backup

(defn select-backup
  "List backups on S3 and select the backup that matches `database`
  and `timestamp`. "
  [system & [database timestamp]]
  (let [database (or database (-> system :db :name))
        timestamp (if timestamp (to-date-time timestamp))]
    (->> (cond->> (find-s3-backups system)
           timestamp (filter #(= timestamp (:time %))))
         (filter #(= database (:db %)))
         (sort-by :time)
         (last))))

(defn create-tmp-db
  "Create a temporary database."
  [{:keys [db] :as system} backup]
  (let [tmp-db (str (java.util.UUID/randomUUID))]
    (shell/exec-checked-script
     "Create temporary database"
     ("export" ~(format "PGPASSWORD=\"%s\"" (:password db)))
     ("createdb"
      "--encoding" "UTF8"
      "--host" ~(:server-name db)
      "--port" ~(:server-port db)
      "--username" ~(:user db)
      "--owner" ~(:user db)
      ~tmp-db))
    (log/infof "Created temporary database %s." tmp-db)
    (assoc backup :tmp-db tmp-db)))

(defn download-data
  "Download the data of `backup` from S3."
  [{:keys [db] :as system} backup]
  (copy-s3-to-local system (:s3-data backup) (:local-data backup))
  (log/infof "Downloaded the data for database %s from S3." (:name db))
  backup)

(defn download-schema
  "Download the schema of `backup` from S3."
  [{:keys [db] :as system} backup]
  (copy-s3-to-local system (:s3-schema backup) (:local-schema backup))
  (log/infof "Downloaded the schema for database %s from S3." (:name db))
  backup)

(defn drop-db
  "Drop the current database."
  [{:keys [db] :as system} backup]
  (shell/exec-checked-script
   "Drop database"
   ("export" ~(format "PGPASSWORD=\"%s\"" (:password db)))
   ("dropdb"
    "--host" ~(:server-name db)
    "--port" ~(:server-port db)
    "--username" ~(:user db)
    "--if-exists"
    ~(:name db)))
  (log/infof "Dropped database %s." (:name db))
  backup)

(defn load-data
  "Load the backup data into the temporary database."
  [{:keys [db] :as system} backup]
  (shell/exec-checked-script
   "Dumping data"
   ("export" ~(format "PGPASSWORD=\"%s\"" (:password db)))
   ("gunzip" "-c" ~(:local-data backup) "|" "psql"
    "--dbname" ~(:tmp-db backup)
    "--host" ~(:server-name db)
    "--port" ~(:server-port db)
    "--username" ~(:user db)))
  (log/infof "Loaded backup data into database %s." (:tmp-db backup))
  backup)

(defn load-schema
  "Load the backup schema into the temporary database."
  [{:keys [db] :as system} backup]
  (shell/exec-checked-script
   "Dumping data"
   ("export" ~(format "PGPASSWORD=\"%s\"" (:password db)))
   ("cat" ~(:local-schema backup) "|" "psql"
    "--dbname" ~(:tmp-db backup)
    "--host" ~(:server-name db)
    "--port" ~(:server-port db)
    "--username" ~(:user db)))
  (log/infof "Loaded backup schema into database %s." (:tmp-db backup))
  backup)

(defn rename-tmp-db
  "Rename the temporary database."
  [{:keys [db] :as system} backup]
  (->> {:command
        (format "ALTER DATABASE \"%s\" RENAME TO \"%s\";"
                (:tmp-db backup) (:name db))}
       (shell/psql (assoc db :name "postgres")))
  (log/infof "Renamed database %s to %s." (:tmp-db backup) (:name db))
  backup)

(defn restore-backup
  "Restore a database backup from S3."
  [config & [database timestamp]]
  (let [system (backup-system config)]
    (if-let [backup (select-backup system database timestamp)]
      (let [started-at (now)]
        (->> (download-schema system backup)
             (download-data system)
             (create-tmp-db system)
             (load-schema system)
             (load-data system)
             (drop-db system)
             (rename-tmp-db system))
        (materialized-views/run (:db config) materialized-views/views)
        (log/infof "Database %s restored in %s minutes."
                   (-> system :db :name)
                   (in-minutes (interval started-at (now)))))
      (log/error "Can't find matching database backup."))))

;; Command line

(defn show-help
  "Show help for `cmd`."
  [cmd]
  (print-help (str "backup " cmd))
  (System/exit 0))

(defmulti run
  "Run the backup `command`."
  (fn [env command & args]
    (keyword command)))

(defmethod run :create [env cmd & args]
  (with-commandline [[opts [cmd & args]] args]
    [[c cleanup "Cleanup local files after completion."]
     [h help "Print this help."]]
    (when (:help opts) (show-help "create [OPTIONS]"))
    (create-backup (config/backup env) (now) opts)))

(defmethod run :list [env cmd & args]
  (with-commandline [[opts [cmd & args]] args]
    [[h help "Print this help."]]
    (when (:help opts) (show-help "list [OPTIONS]"))
    (list-backups (config/backup env))))

(defmethod run :restore [env cmd & args]
  (with-commandline [[opts [cmd & args]] args]
    [[d database "The database name to restore from."]
     [h help "Print this help."]
     [t time "The timestamp to restore from." :time "TIME"]]
    (when (:help opts) (show-help "restore [OPTIONS]"))
    (restore-backup (config/backup env) (:database opts) (:time opts))))

(defmethod run :default [env cmd & args]
  (println (str "Usage: backup [create|list|restore]"))
  (System/exit 1))

(defn -main [& args]
  (let [[cmd & args] args]
    (apply run env cmd args)))
