(ns burningswell.db.comments
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(defn- select-all [db & [opts]]
  (let [{:keys [query page per-page]} opts]
    (select db [:comments.id
                :comments.content
                :comments.visible
                :comments.created-at
                :comments.updated-at
                (as `(json_build_object
                      "spot" (json-embed-spot :spots)
                      "user" (json-embed-user :users))
                    :_embedded)]
      (from :comments)
      (join :users.id :comments.user-id :type :left)
      (join :spots.id :comments.spot-id :type :left)
      (fulltext query :comments.content)
      (paginate page per-page)
      (order-by :comments.spot-id :comments.created-at))))

(s/defn all :- [Comment]
  "Return all comments in `db`."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn by-id :- (s/maybe Comment)
  "Return the country in `db` by `id`."
  [db :- Database id :- s/Num]
  (first @(compose
           (select-all db)
           (where `(= :comments.id (cast ~id :integer))))))

(s/defn by-user :- [Comment]
  "Returns the comments for the given `user`."
  [db :- Database user :- User & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :comments.user-id ~(:id user)))))

(s/defn by-spot :- [Comment]
  "Returns the comments for the given `spot`."
  [db :- Database spot :- Spot & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :comments.spot-id ~(:id spot)))))

(defn- row [comment]
  (assoc (select-keys comment [:content :visible])
         :spot-id (-> comment :_embedded :spot :id)
         :user-id (-> comment :_embedded :user :id)))

(s/defn delete
  "Delete `comment` from `db`."
  [db :- Database comment :- Comment]
  (->> @(sql/delete db :comments
          (where `(= :comments.id
                     ~(:id comment))))
       first :count))

(s/defn insert
  "Insert `comment` into `db`."
  [db :- Database comment]
  (->> @(sql/insert db :comments []
          (values [(row comment)])
          (returning :id))
       first :id (by-id db)))

(s/defn update
  "Update `comment` in `db`."
  [db :- Database comment]
  (->> @(sql/update db :comments
          (row comment)
          (where `(= :comments.id
                     ~(:id comment)))
          (returning :id))
       first :id (by-id db)))

(s/defn sao-lorenco-comment :- (s/maybe Comment)
  "Returns a comment of São Lorenço in Portugal."
  [db :- Database]
  (by-id db 1))
