(ns burningswell.db.spots
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.photos :as photos]
            [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [burningswell.db.weather :as weather]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [geo.postgis :refer [geometry]]
            [no.en.core :refer [parse-integer]]
            [schema.core :as s])
  (:import [org.postgis Point]
           [sqlingvo.db Database]))

(def ^:dynamic *defaults*
  {:distance 3000})

(defn- select-all [db & [opts]]
  (let [distance (or (:distance opts) (:distance *defaults*))
        location (:location opts)]
    (select db [:spots.id
                :spots.name
                (as '(cast :spots.location :geometry) :location)
                :spots.visible
                :spots.created-at
                :spots.updated-at
                (as `(json_build_object
                      "country" (json-embed-country :countries)
                      "region" (json-embed-region :regions)
                      "user" (json-embed-user :users)
                      "photo" ~embedded-photo
                      "time-zone" (json-embed-time-zone :time-zones))
                    :_embedded)]
      (from :spots)
      (join :countries.id :spots.country-id :type :left)
      (join :regions.id :spots.region-id :type :left)
      (join :users.id :spots.user-id :type :left)
      (join :photos.id :spots.photo-id :type :left)
      (join :time-zones.id :spots.time-zone-id :type :left)
      (fulltext (:query opts) :spots.name)
      (paginate (:page opts) (:per-page opts))
      (within-bounding-box :spots.location (:bounding-box opts))
      (when location (within-distance-to :spots.location location distance))
      (if location
        (order-by-distance :spots.location location)
        (order-by :spots.name)))))

(s/defn country-id :- (s/maybe s/Int)
  "Return the country id of `spot`."
  [spot]
  (-> spot :_embedded :country :id))

(s/defn region-id :- (s/maybe s/Int)
  "Return the region id of `spot`."
  [spot]
  (-> spot :_embedded :region :id))

(s/defn user-id :- (s/maybe s/Int)
  "Return the user id of `spot`."
  [spot]
  (-> spot :_embedded :user :id))

(s/defn photo-id :- (s/maybe s/Int)
  "Return the photo id of `spot`."
  [spot]
  (-> spot :_embedded :photo :id))

(s/defn all :- [Spot]
  "Return all spots in `db`."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn around-spot :- [Spot]
  "Returns all spots around `spot`."
  [db :- Database spot :- Spot & [opts]]
  (let [distance (or (:distance opts) (:distance *defaults*))
        location (:location spot)]
    @(compose
      (select-all db (assoc opts :location location))
      (where `(and (not (= :spots.id ~(:id spot)))
                   (st_dwithin
                    (cast :spots.location :geography)
                    (cast ~location :geography)
                    ~(* distance 1000)))))))

(s/defn by-id
  "Return the country in `db` by `id`."
  [db :- Database id :- s/Num]
  (first @(compose
           (select-all db)
           (where `(= :spots.id (cast ~id :integer))))))

(s/defn by-ids :- [Spot]
  "Return all spots in `db` by the list of `ids`."
  [db :- Database ids :- [s/Num] & [opts]]
  @(compose
    (select-all db opts)
    (where `(in :spots.id ~ids))))

(s/defn by-name
  "Return the country in `db` by `name`."
  [db :- Database name :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :spots.name ~name)))))

(s/defn by-location :- (s/maybe Spot)
  "Return the spot in `db` at `location`."
  [db :- Database location :- Point]
  (first @(compose
           (select-all db)
           (where `(= :spots.location ~(geometry location))))))

(s/defn within-distance :- [Spot]
  "Return the spots in `db` within :distance in km to `location`."
  [db :- Database location :- Point distance :- s/Num & [opts]]
  @(compose
    (select-all db (assoc opts :location location))
    (within-distance-to :spots.location location distance)))

(defn- row [spot]
  (assoc (select-keys spot [:name :location :visible])
         :country-id (country-id spot)
         :region-id (region-id spot)
         :user-id (user-id spot)
         :photo-id (photo-id spot)))

(s/defn delete
  "Delete `spot` from `db`."
  [db :- Database spot :- Spot]
  (->> @(sql/delete db :spots
          (where `(= :spots.id
                     ~(:id spot))))
       first :count))

(s/defn in-continent :- [Spot]
  "Returns all spots in `continent`."
  [db :- Database continent :- Continent & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :countries.continent-id ~(:id continent)))))

(s/defn in-country :- [Spot]
  "Returns all spots in `country`."
  [db :- Database country :- Country & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :spots.country-id ~(:id country)))))

(s/defn in-region :- [Spot]
  "Returns all spots in `region`."
  [db :- Database region :- Region & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :spots.region-id ~(:id region)))))

(s/defn insert
  "Insert `spot` into `db`."
  [db :- Database spot]
  (->> @(sql/insert db :spots []
          (values [(row spot)])
          (returning :id))
       first :id (by-id db)))

(s/defn update
  "Update `spot` in `db`."
  [db :- Database spot]
  (->> @(sql/update db :spots (row spot)
                    (where `(= :spots.id
                               ~(:id spot)))
                    (returning :id))
       first :id (by-id db)))

(s/defn save :- Spot
  "Insert `spot` into `db`."
  [db :- Database spot]
  (or (update db spot)
      (insert db spot)))

(s/defn banzai-pipeline :- (s/maybe Spot)
  "Returns the spot Banzai Pipline, Hawaii, United States."
  [db :- Database]
  (by-id db 5))

(s/defn menakoz :- (s/maybe Spot)
  "Returns the spot Meñakoz, Spain."
  [db :- Database]
  (by-id db 1))

(s/defn mundaka :- (s/maybe Spot)
  "Returns the spot Mundaka, Spain."
  [db :- Database]
  (by-id db 2))

(s/defn padang-padang :- (s/maybe Spot)
  "Returns the spot Padang Padang, Indonesia."
  [db :- Database]
  (by-id db 3))

(s/defn uluwatu :- (s/maybe Spot)
  "Returns the spot Uluwatu, Indonesia."
  [db :- Database]
  (by-id db 4))

(s/defn sao-lorenco :- (s/maybe Spot)
  "Returns the spot São Lorenço, Portugal."
  [db :- Database]
  (by-id db 6))

(s/defn assoc-photo :- [Spot]
  "Assoc the photo and their images to all `spots`."
  [db :- Database spots :- [Spot]]
  (let [photos (zip-by-id (photos/by-spots db spots {:images true}))]
    (map #(->> (get photos (-> % :_embedded :photo :id))
               (assoc-in % [:_embedded :photo]))
         spots)))

(s/defn assoc-current-weather :- [Spot]
  "Assoc the current weather forecasts to `spots`."
  [db :- Database spots :- [Spot] & [opts]]
  (let [weather (weather/current-weather-by-spots db spots opts)]
    (map #(assoc-in % [:_embedded :weather]
                    (get weather (:id %)))
         spots)))

(s/defn without-a-photo :- [Spot]
  "Return all spots that don't have a photo."
  [db :- Database & [opts]]
  @(compose
    (select-all db opts)
    (where `(is-null :spots.photo-id))))

(s/defn set-timezone
  "Set the timezone of all spots."
  [db :- Database & [opts]]
  (->> @(sql/update db :spots
          {:time-zone-id :t.id}
          (from (as (select db [:time-zones.id (as :spots.id :spot-id)]
                      (from :time-zones)
                      (join :spots
                            '(on (st_intersects
                                  :spots.location
                                  :time-zones.geom))))
                    :t))
          (where '(= :spots.id :t.spot-id)))
       first :count))
