(ns burningswell.db.roles
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.util :refer :all]
            [datumbazo.core :refer :all]))

(derive ::admin ::surfer)
(derive ::editor ::surfer)

(defn role-keyword [role]
  (keyword (str "burningswell.db.roles/" (name role))))

(deftable roles
  "The roles database table."
  (column :id :serial :primary-key? true)
  (column :name :citext :not-null? true :unique? true)
  (column :description :citext :not-null? true :unique? true)
  (column :created-at :timestamp-with-time-zone
          :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(defquery roles
  "Returns all roles."
  [db & [opts]]
  (let [{:keys [query page per-page]} opts]
    (select db (remove :hidden? (columns roles-table))
      (from :roles)
      (fulltext query :roles.name :roles.description)
      (paginate page per-page))))

(defn admin
  "Returns the admin role."
  [db] (role-by-id db 1))

(defn editor
  "Returns the editor role."
  [db] (role-by-id db 2))

(defn surfer
  "Returns the surfer role."
  [db] (role-by-id db 3))

(defn role-by-param
  "Returns the role by param."
  [db param] (role-by-id db param))

(defn role-by-req
  "Returns the role for the Pedestal request `req`."
  [req]
  (role-by-param (:db req) (-> req :path-params :id)))

(defquery roles-by-user
  "Returns the roles for the given `user`."
  [db user & [opts]]
  (compose
   (roles* db opts)
   (join :roles-users.role-id :roles.id)
   (where `(= :roles-users.user-id ~(:id user)))))

(defn create-role
  "Create a new role."
  [db role]
  (let [row (insert-role db role)]
    (role-by-id db (:id row))))
