(ns burningswell.db.users
  (:refer-clojure :exclude [distinct find group-by update])
  (:require [burningswell.db.roles :as roles]
            [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [clojure.set :refer [rename-keys]]
            [clojure.spec :as sp]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [no.en.core :refer [compact-map parse-integer]]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(defn- select-all [db & [opts]]
  (let [{:keys [location query page per-page]} opts]
    (select db [:users.id
                :users.username
                :users.email
                :users.name
                :users.first-name
                :users.last-name
                (as '(cast :users.location :geometry) :location)
                :users.locale
                :users.facebook-id
                :users.facebook-url
                :users.google-id
                :users.google-url
                :users.twitter-id
                :users.twitter-url
                :users.created-at
                :users.updated-at
                (as `(json_build_object
                      "country" (json-embed-country :countries)
                      "region" (json-embed-region :regions)
                      "settings" ~(embedded-user-settings :user-settings))
                    :_embedded)]
      (from :users)
      (join :countries '(on (= :countries.id :users.country-id)) :type :left)
      (join :regions '(on (= :regions.id :users.region-id)) :type :left)
      (join :user-settings
            '(on (= :user-settings.user-id :users.id))
            :type :left)
      (fulltext query :users.username :users.name)
      (if location
        (order-by-distance :location location)
        (order-by :username))
      (paginate page per-page))))

(s/defn add-to-role :- s/Any
  "Add the `role` to `user`."
  [db :- Database user :- User role :- Role]
  @(select db [`(add-user-to-role ~(:id user) ~(:id role))]))

(s/defn all :- [User]
  "Returns all users."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn by-email :- (s/maybe User)
  "Return the user in `db` by `email`."
  [db :- Database email :- (s/maybe s/Str)]
  (when email
    (first @(compose
             (select-all db)
             (where `(= :users.email ~email))))))

(s/defn by-id :- (s/maybe User)
  "Return the user in `db` by `id`."
  [db :- Database id :- s/Int]
  (first @(compose
           (select-all db)
           (where `(= :users.id (cast ~id :integer))))))

(s/defn by-google-id :- (s/maybe User)
  "Return the user in `db` by `google-id`."
  [db :- Database google-id :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :users.google-id ~google-id)))))

(s/defn by-facebook-id :- (s/maybe User)
  "Return the user in `db` by `facebook-id`."
  [db :- Database facebook-id :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :users.facebook-id ~facebook-id)))))

(s/defn by-twitter-id :- (s/maybe User)
  "Return the user in `db` by `twitter-id`."
  [db :- Database twitter-id :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :users.twitter-id ~twitter-id)))))

(s/defn by-username :- (s/maybe User)
  "Return the user in `db` by `username`."
  [db :- Database username :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :users.username ~username)))))

(s/defn encrypt-password :- s/Str
  "Encrypt `password` with the Blowfish cipher."
  [db :- Database password :- s/Str]
  (-> @(select db [(as `(crypt ~password (gen_salt "bf")) :crypted-password)])
      first :crypted-password))

(s/defn assoc-crypted-password
  "If `user` has a :password key, encrypt it and add it
  as :crypted-password to `user`, otherwise return `user`."
  [db :- Database user]
  (if-let [password (:password user)]
    (assoc user :crypted-password (encrypt-password db password))
    user))

(defn- row [continent]
  (assoc (select-keys continent
                      [:username :email :name :first-name :last-name
                       :location :locale :crypted-password
                       :google-id :google-url
                       :facebook-id :facebook-url
                       :twitter-id :twitter-url])
         :country-id (-> continent :_embedded :country :id)
         :region-id (-> continent :_embedded :region :id)))

(s/defn delete
  "Delete `user` from `db`."
  [db :- Database user :- User]
  (->> @(sql/delete db :users
                    (where `(= :users.id
                               ~(:id user))))
       first :count))

(s/defn insert
  "Insert `user` into `db`."
  [db :- Database user]
  (let [user (assoc-crypted-password db user)]
    (->> @(sql/insert db :users []
                      (values [(row user)])
                      (returning :id))
         first :id (by-id db))))

(s/defn update
  "Update `user` in `db`."
  [db :- Database user]
  (let [user (assoc-crypted-password db user)]
    (->> @(sql/update db :users
                      (row user)
                      (where `(= :users.id ~(:id user)))
                      (returning :id))
         first :id (by-id db))))

(s/defn merge-users :- s/Any
  "Merge `source-user` into `target-user`."
  [db :- Database source-user :- User target-user :- User]
  (assert (not= (:id source-user) (:id target-user))
          "Source and target user must be different.")
  (with-transaction [db db]
    (doseq [table [:comments :photos :sessions :spots :ratings :roles-users]]
      @(sql/update db table {:user-id (:id target-user)}
                   (where `(= :user-id ~(:id target-user)))))
    (delete db source-user)
    (update db (merge target-user (compact-map (dissoc source-user :id))))))

(s/defn in-continent :- [User]
  "Returns all users in `continent`."
  [db :- Database continent & [opts]]
  (let [{:keys [query page per-page]} opts]
    @(compose
      (select-all db opts)
      (where `(= :countries.continent-id ~(:id continent)))
      (fulltext query :users.login :users.username)
      (order-by-distance :location (:location opts))
      (paginate page per-page))))

(s/defn in-country :- [User]
  "Returns all users in `country`."
  [db :- Database country & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :users.country-id ~(:id country)))))

(s/defn in-region :- [User]
  "Returns all users in `region`."
  [db :- Database region & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :users.region-id ~(:id region)))))

(s/defn authenticate-password :- (s/maybe User)
  "Try to authenticate `user` with a password."
  [db :- Database user]
  (let [login (first (remove nil? (map (or user {})
                                       [:login :username :email])))]
    (first @(compose
             (select-all db)
             (where `(and (or (= :username ~login)
                              (= :email ~login))
                          (= :crypted-password
                             (crypt ~(:password user) :crypted-password))))))))

(s/defn authenticate
  "Rerturns a friend compatible authentication map."
  [db :- Database user]
  (if-let [user (authenticate-password db user)]
    (->> (roles/roles-by-user db user)
         (map #(keyword (str "burningswell.db.roles/" (:name %1))))
         (set)
         (assoc user :roles))))

(defn facebook-user
  "Convert the Facebook `profile` into a user."
  [profile]
  {:email (:email profile)
   :facebook-id (:id profile)
   :facebook-url (:link profile)
   :first-name (:first-name profile)
   :gender (:gender profile)
   :last-name (:last-name profile)
   :locale (:locale profile)
   :name (:name profile)
   :username (:username profile)})

(defn google-user
  "Convert the Google `profile` into a user."
  [profile]
  {:first-name (:given-name profile)
   :gender (:gender profile)
   :google-id (:id profile)
   :google-url (:link profile)
   :google-verified (:verified-email profile)
   :email (:email profile)
   :last-name (:family-name profile)
   :locale (:locale profile)
   :name (:name profile)
   :username (:name profile)})

(defn twitter-user
  "Convert the Twitter `profile` into a user."
  [profile]
  {:twitter-id (:id profile)
   :username (:screen-name profile)
   :name (:name profile)
   :twitter-url (format "https://twitter.com/%s" (:screen-name profile))})

(s/defn insert-facebook-user :- User
  "Insert the Facebook profile and return a user."
  [db :- Database profile]
  (let [user (insert db (facebook-user profile))]
    (add-to-role db user (roles/surfer db))
    user))

(s/defn insert-google-user :- User
  "Insert the Google profile and return a user."
  [db :- Database profile]
  (insert db (google-user profile)))

(s/defn insert-twitter-user :- User
  "Insert the Twitter profile and return a user."
  [db :- Database profile]
  (insert db (twitter-user profile)))

(s/defn update-facebook-user :- (s/maybe User)
  "Update the Facebook profile and return a user."
  [db :- Database profile]
  (if-let [user (or (by-email db (:email profile))
                    (by-facebook-id db (:id profile)))]
    (update db (merge user (facebook-user profile)))))

(s/defn update-facebook-user :- (s/maybe User)
  "Update the Facebook profile and return a user."
  [db :- Database profile]
  (let [user-1 (by-email db (:email profile))
        user-2 (by-facebook-id db (:id profile))
        profile-attrs (facebook-user profile)]
    (cond
      (and user-1 user-2 (not= (:id user-1) (:id user-2)))
      (update db (merge (merge-users db user-2 user-1) profile-attrs))
      (or user-1 user-2)
      (update db (merge user-1 user-2 profile-attrs))
      :else nil)))

(s/defn update-google-user :- (s/maybe User)
  "Update the Google profile and return a user."
  [db :- Database profile]
  (if-let [user (or (by-email db (:email profile))
                    (by-google-id db (:id profile)))]
    (update db (merge user (google-user profile)))))

(s/defn update-twitter-user :- (s/maybe User)
  "Update the Twitter profile and return a user."
  [db :- Database profile]
  (if-let [user (by-twitter-id db (str (:id profile)))]
    (update db (merge user (twitter-user profile)))))

(s/defn save-facebook-user :- User
  "Save the Twitter profile and return a user."
  [db :- Database profile]
  (or (update-facebook-user db profile)
      (insert-facebook-user db profile)))

(s/defn save-google-user :- User
  "Save the Twitter profile and return a user."
  [db :- Database profile]
  (or (update-google-user db profile)
      (insert-google-user db profile)))

(s/defn save-twitter-user :- User
  "Save the Twitter profile and return a user."
  [db :- Database profile]
  (or (update-twitter-user db profile)
      (insert-twitter-user db profile)))

(s/defn by-access-token :- (s/maybe User)
  "Returns the user by `access-token`."
  [db :- Database access-token]
  (first @(compose
           (select-all db)
           (join :oauth.access-tokens.user-id :users.id)
           (where `(and (= :oauth.access-tokens.access-token ~access-token)
                        (is-null :oauth.access-tokens.revoked-at)
                        (or (is-null :oauth.access-tokens.expires-at)
                            (< (now) :oauth.access-tokens.expires-at)))))))

(def safe-columns
  [:_embedded
   :created-at
   :first-name
   :id
   :last-name
   :name
   :region-id
   :updated-at
   :username])

(defn safe-user [current user]
  (let [user (dissoc user :crypted-password)]
    (if (= (:id current) (:id user))
      user
      (select-keys user safe-columns))))

(s/defn root :- (s/maybe User)
  "Returns the user root."
  [db :- Database]
  (by-id db 1))

(s/defn roman :- (s/maybe User)
  "Returns the user roman."
  [db :- Database]
  (by-id db 2))

(s/defn bodhi :- (s/maybe User)
  "Returns the user bodhi."
  [db :- Database]
  (by-id db 3))

(s/defn roach :- (s/maybe User)
  "Returns the user roach."
  [db :- Database]
  (by-id db 4))

(s/defn stormrider :- (s/maybe User)
  "Returns the user stormrider."
  [db :- Database]
  (by-id db 5))

(s/defn email-available? :- s/Bool
  "Returns the true if `email` is available, otherwise false."
  [db :- Database email :- s/Str]
  (nil? (by-email db email)))

(s/defn username-available? :- s/Bool
  "Returns the true if `username` is available, otherwise false."
  [db :- Database username :- s/Str]
  (nil? (by-username db username)))

(s/defn has-role-name? :- s/Bool
  "Returns the true if `user` has `role` name."
  [db :- Database user :- User role :- s/Keyword]
  (->> @(select db [:roles.id]
          (from :roles-users)
          (join :roles.id :roles-users.role-id)
          (where `(and (= :roles-users.user-id ~(:id user))
                       (= :roles.name ~(name role)))))
       empty? not))

(s/defn has-role? :- s/Bool
  "Returns the true if `user` has `role`."
  [db :- Database user :- User role :- Role]
  (->> @(select db [:roles.id]
          (from :roles-users)
          (join :roles.id :roles-users.role-id)
          (where `(and (= :roles-users.user-id ~(:id user))
                       (= :roles-users.role-id ~(:id role)))))
       empty? not))

(s/defn find :- (s/maybe User)
  "Find a user in `db` by `identifier`, which can be an id, the
  username or email address."
  [db :- Database identifier]
  (cond
    (number? identifier)
    (by-id db identifier)
    (and (string? identifier)
         (re-matches #"\d+" identifier))
    (by-id db (Long/parseLong identifier))
    (string? identifier)
    (or (by-username db identifier)
        (by-email db identifier))))

(defn email-available?
  "Returns true if `email` is available, otherwise false."
  [db email]
  (-> @(select db [1]
         (from :users)
         (where `(= (lower :users.email) (lower ~email))))
      (empty?)))

(sp/fdef email-available?
  :args (sp/cat :db db? :email string?)
  :ret boolean?)

(defn username-available?
  "Returns true if `username` is available, otherwise false."
  [db username]
  (-> @(select db [1]
         (from :users)
         (where `(= (lower :users.username) (lower ~username))))
      (empty?)))

(sp/fdef username-available?
  :args (sp/cat :db db? :username string?)
  :ret boolean?)

(defn settings
  "Returns the settings for `user` from `db`."
  [db user]
  (->> @(select db [:*]
          (from :user-settings)
          (where `(= :user-settings.user-id ~(:id user))))
       (first)))

(defn save-settings
  "Save the `settings` for `user` to `db`."
  [db settings]
  (->> @(sql/insert db :user-settings [:user-id :units]
          (values [(select-keys settings [:user-id :units])])
          (on-conflict [:user-id]
            (do-update {:units :EXCLUDED.units}))
          (returning :*))
       first))
